// timer_interrupt.c : the low level code run on a fixed rate timer interrupt
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------

// The timer interrupt is used for all the polling tasks: 
//   counting encoder position
//   sampling the IMU signals with the A/D
//   keeping track of time
//
// This is a replacement for the libLPC2xxx standard timer interrupt,
// which should be configured as inactive as follows:
//   #define CONFIG_INSTALL_DEFAULT_TIMER_INTERRUPT 0

#include <libLPC2xxx.h>
#include <interrupt_handlers.h>
#include <timing.h>
#include <errcodes.h>

#include <timer_interrupt.h>
#include <encoder_counter.h>
#include <imu.h>

/****************************************************************/
#if CONFIG_TIMER_INTERRUPT_REPORTS_LOAD
// holds the timer counter value near the end of the timer interrupt
volatile unsigned int timer_interrupt_load_indicator;  
volatile unsigned int timer_interrupt_max_load = 0;
#endif

/****************************************************************/
// Timer 1 Interrupt Service Routine
// Permissible values for the "interrupt" attribute parameter are:
// IRQ, FIQ, SWI, ABORT and UNDEF. This behavior changed with GCC 3.4.4.

void __attribute__((interrupt ("IRQ"))) timer_interrupt_handler(void)
{
  // decode the encoder inputs
  encoder_counter_poll();

  // update the A/D conversions for the IMU
  imu_poll();

  // Clear timer interrupt by writing a one for the MR0 bit.
  TIMER1.IR = TIMER_IR_MR0_MASK;   

  // update the time count value
  timerclock++;

#if CONFIG_TIMER_INTERRUPT_REPORTS_LOAD
  // save the timer counter value near the end of the timer interrupt
  timer_interrupt_load_indicator = TIMER1.TC;
  if ( timer_interrupt_load_indicator > timer_interrupt_max_load ) 
    timer_interrupt_max_load = timer_interrupt_load_indicator;
#endif

  // A write must be performed to the VIC Vector Address Register to
  // update the VIC priority hardware.
  VIC.VectAddr = 0;
}

/****************************************************************/

// Install the timer interrupt.
int
init_timer_interrupt(void)
{
  // Set up the VIC for the interrupt
  VIC.VectAddr0   = (unsigned long) timer_interrupt_handler;
  VIC.VectCntl0 = VIC_VECTCNTL_ENABLED_MASK | VIC_TIMER1_CHANNEL;
  VIC.IntEnable = VIC_TIMER1_MASK;

  // Set up the timer
  TIMER1.TCR   = 0;    // turn it off
  TIMER1.TC    = 0;    // clear so we don't miss first match
  TIMER1.PR    = 0;    // prescale of 0 -> tick rate == sysclk
  TIMER1.PC    = 0;    // clear prescale counter for no particular reason

  // set the divisor for the timer interrupt
  TIMER1.MR0 = TIMER_INTERRUPT_DIVISOR;
  
  // Reset and interrupt on match to generate a periodic interrupt.
  TIMER1.MCR = TIMER_MCR_IMR0_MASK | TIMER_MCR_RMR0_MASK;  // RMR0 = IMR0 = 1

  // Start the timer interrupt clock.
  TIMER1.TCR = TIMER_TCR_ENABLE_MASK;

  return ERRNOERROR;
}

/****************************************************************/
