// example-controller.c : run a Balancer controller on a host using a serial port for I/O
// This file is part of ArtLPC. 
// This file is placed into the public domain.

#include <stdio.h>

#include <libstd.h>
#include <libLPC2xxx.h>
#include <posix_serial_port.h>
#include <string_port.h>
#include <errcodes.h>

// for my G5:
// #define SERIAL_PORT "/dev/cu.USA19QWb12P1.1"
// for my laptop, using the rearward serial port
#define SERIAL_PORT "/dev/cu.USA19QW1b1P1.1"

/****************************************************************/
int main (int argc, char **argv)
{
  FILE *in, *out;
  int err;

  printf("opening serial port to communicate with robot.\n");
  err = open_posix_serial_port_in_raw_mode( SERIAL_PORT, 115200, &in, &out );

  if (err) {
    printf("failed to open serial port, error %d\n", err);
    exit(1);
  }

  // read sensor data and emit controls
  for(;;) {
    int leftpos, rightpos, xaccel, yaccel, zaccel, pitchvel, rollvel;
    int i, c;
#define MAXSENSORLENGTH 200
    char linebuf[MAXSENSORLENGTH+1];
    int numargs;
    int uleft, uright;

    // first read a line of ASCII sensor data, ignoring overruns
    for( i = 0 ; ; ) {
      c = fgetc(in);
      if ( c == EOF ) goto done;
      else if ( c == '\n' ) break;
      else if ( i < MAXSENSORLENGTH ) linebuf[i++] = c;
    }
    linebuf[i] = 0;

    // if the line begins with '#', it is debugging output
    if (linebuf[0] == '#') {
      printf( "DBG: %s\n", linebuf );
      continue;
    }

    // parse the sensor data line
    numargs = sscanf( linebuf, "L%dR%dX%dY%dZ%dP%dQ%d",
		      &leftpos, &rightpos, &xaccel, &yaccel, &zaccel, &pitchvel, &rollvel );
    if ( numargs != 7 ) {
      printf("only %d args received, ignoring line '%s'.\n", numargs, linebuf);
      continue;
    } else {
      printf("leftpos: %d rightpos: %d xaccel: %d yaccel: %d zaccel: %d pitchvel: %d rollvel %d --> ",
	     leftpos, rightpos, xaccel, yaccel, zaccel, pitchvel, rollvel );
    }

    // compute some trivial control function
    uleft  = (int) (0.01 * (0 - leftpos));  // pure proportional gain
    uright = (int) (0.01 * (0 - rightpos)); // pure proportional gain
    
    // limit the command values
    if (uleft < -127) uleft = -127;
    else if (uleft > 127) uleft = 127;
    if (uright < -127) uright = -127;
    else if (uright > 127) uright = 127;

    // and send a command packet
    fprintf(out, "L%dR%d\n", uleft, uright);
    printf("ul: %d ur: %d\n", uleft, uright);
    fflush(out);
  }

 done:
  printf("control loop exited.\n");
  fclose(in);
  fclose(out);
  return 0;
}
/****************************************************************/


