// string_port.c : interface to fixed sized string buffer using port abstraction
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------
// A simple fixed-sized buffer port.  This doesn't automatically
// append a C '\0' string terminator, since a length field is included
// in the port structure.  For simultaneous reading and writing it
// would be better to create a fifo_port.

#include <libstd.h>
#include <port.h>
#include <string_port.h>
#include <errcodes.h>

/****************************************************************/
// Private definitions.

/****************************************************************/
static int string_port_write_char( struct port_t *port_ptr, int c)
{
  string_port *p = (string_port *) port_ptr;

  if ( p->position >= p->buffer_size ) {
    return -ERRNOSPACE;
  }

  p->buffer[ p->position++ ] = c;
  if ( p->characters < p->position ) p->characters = p->position;
  return ERRNOERROR;
}

/****************************************************************/
static int string_port_read_char( struct port_t *port_ptr )
{
  string_port *p = (string_port *) port_ptr;

  if ( p->position >= p->characters )  return EOF;
  else return p->buffer[ p->position++ ];
}

static int string_port_peek_char( struct port_t *port_ptr )
{
  string_port *p = (string_port *) port_ptr;
  if ( p->position >= p->characters )  return EOF;
  else return p->buffer[ p->position ];
}

/****************************************************************/
const static struct port_class_t string_port_class = {
  // use a C99 notation for selectively initializing structure elements
  .read_char  = string_port_read_char,  // int  (*read_char)(struct port_t *p);
  .peek_char  = string_port_peek_char,  // int  (*peek_char)(struct port_t *p);
  .write_char = string_port_write_char, // int  (*write_char)(struct port_t *p, int c);
};

/****************************************************************/
int string_port_init( string_port *p, void *buffer, size_t chars, unsigned mode )
{
  generic_port_init( &p->interface, &string_port_class, mode );

  // write ports start out empty
  if ( mode & PORT_FLAGS_WRITE ) p->characters = 0;

  // read ports start out full
  if ( mode & PORT_FLAGS_READ )  p->characters = chars;

  p->buffer = (unsigned char *) buffer;
  p->buffer_size = chars;
  p->position = 0;

  return ERRNOERROR;
}

/****************************************************************/
int  
string_port_zero_terminate( string_port *p )
{
  if ( !(p->interface.flags & PORT_FLAGS_WRITE) ) return -ERRREADONLY;

  if ( p->characters >= p->buffer_size ) {
    p->buffer[ p->buffer_size-1 ] = 0;
    return -ERRIOERROR;
  } else {
    p->buffer[ p->characters ] = 0;
    return ERRNOERROR;
  }
}

/****************************************************************/
