// watchdog.c : interface to the hardware watchdog timer
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------
#include <libLPC2xxx.h>

// Reset the watchdog timer and start the watchdog.
void watchdog_feed(void)
{
  register unsigned int saved_cpsr asm ("r0");

  // The Philips manual doesn't say it, but interrupts should be
  // disabled during feeding.  What it says is that no other accesses
  // to the watchdog register space are allowed during the feed
  // sequence, however, it appears that an interrupt during the
  // sequence triggers the watchdog.  It is not clear whether it is
  // caused by the delay of the handler running or the handler
  // accessing other peripherals.

  // disable FIQ and IRQ interrupts
  asm("mrs r0, cpsr");      // copy the CPSR into R0
  asm("orr r1, r0, #0xc0"); // set the interrupt mask bits to disable interrupts
  asm("msr cpsr_c, r1");    // modify the CPSR

  WDT.WDFEED = 0xaa;        // this sequence resets the watchdog countdown timer
  WDT.WDFEED = 0x55;

  // restore previous interrupt state
  asm("msr cpsr_c, r0");    // restore the original CPSR value including interrupt flags
}

#if CONFIG_WATCHDOG_INTERVAL
void watchdog_init(void)
{
  // The watchdog clock is the PCLK divided by four.  The minimum
  // watchdog divisor is 0xff.  WATCHDOG_INTERVAL might be a constant floating
  // point number, which can be cast to an integer during compilation.
  WDT.WDTC = (unsigned int) (CONFIG_WATCHDOG_INTERVAL * (PCLK/4));

  // Enable the watchdog and enable generating a reset or interrupt.
  // If the WDRESET bit is zero then watchdog timeout generates an
  // interrupt instead of a reset.

#if CONFIG_WATCHDOG_GENERATES_INTERRUPT
  WDT.WDMOD = WDMOD_WDEN_MASK | 0;
#else
  WDT.WDMOD = WDMOD_WDEN_MASK | WDMOD_WDRESET_MASK;
#endif

  // Start the watchdog by issuing a feed sequence.
  watchdog_feed();
}
#endif // CONFIG_WATCHDOG_INTERVAL
