// timing.c : utility routines to compute time intervals
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------
#include <libstd.h>
#include <interrupt_handlers.h>
#include <timing.h>
#include <watchdog.h>
#include <libLPC2xxx.h>
#include <errcodes.h>
#include <LED_indicators.h>

// The timerclock value is 32 bits.  If treated as a unsigned integer,
// it wraps around at 0xffffffff -> 0x00000000.  At a 44.1 kHz update
// rate this will happen a little less than once a day, so wraparound
// must be handled correctly.  

// If the current and previous counter values are subtracted as
// unsigned ints, the differencing will work correctly even at the
// wraparound, given the overflow properties of 32 bit arithmetic.

// e.g. current - previous
//          100 - 99          = 1          
//            0 - 0xffffffff  = 1
//
// maximum difference:
//   0xffffffff - 0           = 0xffffffff
//            0 - 1           = 0xffffffff

/****************************************************************/
// The timer counter variable is only actually declared if the timer
// interrupt which drives it is enabled; that way, any code which
// depends on this will compile but fail to link until the interrupt
// is enabled.

#if CONFIG_TIMER_INTERRUPT_RATE
volatile unsigned int timerclock;  
#endif

/****************************************************************/

void timersleep( unsigned int counts )
{
  unsigned int now, last;
  unsigned int elapsed = 0;
  last = timerclock;
  do {
    now = timerclock;       // sample the volatile value
    elapsed += now - last;  // this handles wraparound correctly
    last = now;

    // feed the watchdog if necessary; by definition this is a programmed sleep
#if CONFIG_WATCHDOG_INTERVAL
    watchdog_feed();
#endif

    // blink while sleeping if possible
#if CONFIG_CPU_LED_MASK
    if ( timerclock & 0x0800 ) CPU_LED_on();
    else                       CPU_LED_off();
#endif

  } while ( elapsed < counts );
}

/****************************************************************/
int
elapsed_timer_init( elapsed_timer *timer )
{
  if ( timer == NULL ) return -ERRNULLPTR;
  timer->elapsed = 0;
  timer->last_clock = timerclock;
  return ERRNOERROR;
}

unsigned
elapsed_timer_update( elapsed_timer *timer )
{
  unsigned now = timerclock;
  if ( timer == NULL ) return 0xffffffff;
  timer->elapsed += now - timer->last_clock;
  timer->last_clock = now;
  return timer->elapsed;
}

/****************************************************************/
