// timer_interrupt.c : default timer interrupt handler
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------
// The timer interrupt can be used to poll lots of devices in addition
// to keeping track of elapsed time.

#include <libLPC2xxx.h>
#include <interrupt_handlers.h>
#include <timing.h>

#if TIMER_INTERRUPT_UPDATES_POLLED_SERIAL_FIFOS
#include <polled_serial.h>
#endif

/****************************************************************/
#if CONFIG_TIMER_INTERRUPT_REPORTS_LOAD
// holds the timer counter value near the end of the timer interrupt
volatile unsigned int timer_interrupt_load_indicator;  
volatile unsigned int timer_interrupt_max_load = 0;
#endif

/****************************************************************/
// Timer 1 ISR
// Permissible values for the "interrupt" attribute parameter are:
// IRQ, FIQ, SWI, ABORT and UNDEF. This behavior changed with GCC 3.4.4.

void __attribute__((interrupt ("IRQ"))) timer_interrupt_handler(void)
{

#if TIMER_INTERRUPT_UPDATES_POLLED_SERIAL_FIFOS
  // run these at a fraction of the timer speed
  if ( timerclock & 1) serial_output_poll();
  else serial_input_poll();
#endif

  // other polling activity can go here

  // Clear timer interrupt by writing a one for the MR0 bit.
  TIMER1.IR = TIMER_IR_MR0_MASK;   

  // update the time count value
  timerclock++;

#if CONFIG_TIMER_INTERRUPT_REPORTS_LOAD
  // save the timer counter value near the end of the timer interrupt
  timer_interrupt_load_indicator = TIMER1.TC;
  if ( timer_interrupt_load_indicator > timer_interrupt_max_load ) 
    timer_interrupt_max_load = timer_interrupt_load_indicator;
#endif

  // A write must be performed to the VIC Vector Address Register to
  // update the VIC priority hardware.
  VIC.VectAddr = 0;
}
/****************************************************************/
