// startup.S : interrupt vector, reset handler, and stack initialization
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------
@ Define the interrupt vector table and basic machine initialization.

@ Note that this has been coded as a jump table since all the routines are
@ trivial and in the same file. Each of these could also have been an indirect
@ jump as follows:   LDR PC, =handler_addr

.code 32			@ normal ARM7 32 bit code (not thumb mode)
.section .interrupt, "a"	@ allocatable new code section for interrupt vectors, to be placed at address 0 by the linker script
.extern default_FIQ_handler	@ fast interrupt handler in C code
.global _reset_vector		@ this is a global so the linker can be directed not to strip this section

_reset_vector:	

	@ ARM7 exception table, beginning at address 0x00000000	
	B _start			@ reset
	B Undefined_Handler		@ undefined instruction
	B SWI_Handler			@ software interrupt
	B Prefetch_Handler		@ prefetch abort
	B Abort_Handler			@ data abort

	@ At 0x14 the user should insert a signature (checksum). This
	@ signature enables the bootloader to determine if there is valid
	@ user code in the Flash. Currently most of the Flash programming
	@ tools (debuggers and ISP utility) have this feature built-in so
	@ the end user need not worry about it. If the tool does not provide
	@ this feature then the value has to be computed manually and has to
	@ be inserted at 0x14. Details on computation of checksum could be
	@ found in the Flash programming chapter in the LPC2104/5/6 User
	@ Manual.

	NOP       		@ 0x00000014, reserved interrupt entry, checksum location for boot loader
				
	LDR PC, [PC, #-0xFF0] 	@ 0x0000018: IRQ handler, load IRQ vector from VIC register 0xfffff030
				@ note how this uses PC-relative addressing, since PC is now 0x00000020
	
	LDR PC,=default_FIQ_handler	@ 0x000001c: FIQ handler

	@ end of ARM7 exception table

@ =========================================================
@ Exception Handlers
@ The following dummy handlers just stall until the watchdog can reset the CPU.

Undefined_Handler:
        B         Undefined_Handler

SWI_Handler:
	B         SWI_Handler

Prefetch_Handler:
	B         Prefetch_Handler

Abort_Handler:
	B         Abort_Handler

FIQ_Handler:
	B         FIQ_Handler

@ ================================================================
@ Entry point for basic machine initialization.  Set up stack
@ pointers for supervisor mode (i.e., normal programs) and
@ interrupt mode.

.extern __main		@ the C function __main, defined in main.c 
.extern _stack_top	@ linker defined symbol for placing the program stack
.extern _IRQ_stack_top	@ linker defined symbol for placing the interrupt stack
.extern _FIQ_stack_top	@ linker defined symbol for placing the fast interrupt stack

.section .text		@ this is an ordinary text segment, and could in principle be linked anywhere

_start:
	@ The ARM7TDMI starts in Supervisor mode (0x13) after reset, with FIQ and IRQ interrupts disabled.
	MSR cpsr_c,#0x13	@ Enable interrupts by clearing I and F without changing mode.
	LDR SP,=_stack_top	@ Set SP for Supervisor mode.
	
	@ Setting up SP for IRQ mode. Change mode to IRQ before setting SP_irq
	@ and then move back again to Supervisor mode.  To clarify: the ARM
	@ switches to a banked SP (R13) in IRQ mode.
	
	MRS R0, CPSR		@ save CPSR in R0
	BIC R1, R0, #0x1F	@ R1 = R0 & ~0x1f (CPSR copy with mode bits cleared, BIC is bit clear)
	ORR R1, R1, #0x12	@ R1 |= MODE_IRQ  (set mode bits to IRQ mode)
	MSR cpsr_c, R1		@ set CPSR from R1 to enter IRQ mode
	LDR SP, =_IRQ_stack_top	@ now in IRQ mode, set SP_irq to constant value
	MSR cpsr_c, R0		@ restore CPSR from R0 to restore Supervisor mode

	@ Setting up SP for FIQ mode. Change mode to FIQ before setting SP_fiq
	@ and then move back again to Supervisor mode.  To clarify: the ARM
	@ switches to a banked SP (R13) in FIQ mode.
	
	MRS R0, CPSR		@ save CPSR in R0
	BIC R1, R0, #0x1F	@ R1 = R0 & ~0x1f (CPSR copy with mode bits cleared, BIC is bit clear)
	ORR R1, R1, #0x11	@ R1 |= MODE_FIQ  (set mode bits to FIQ mode)
	MSR cpsr_c, R1		@ set CPSR from R1 to enter FIQ mode
	LDR SP, =_FIQ_stack_top	@ now in FIQ mode, set SP_fiq to constant value
	MSR cpsr_c, R0		@ restore CPSR from R0 to restore Supervisor mode

	@ Jump to C code
	LDR lr, =__main		@ branch to C function __main()
	MOV pc, lr		@ with the side effect that link register holds the address of __main(),
				@ which will cause __main() to be re-entered if it ever returns

@ ================================================================
