// posix-serial-port.c : interface to a POSIX standard serial port on a host computer
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------

#include <stdio.h>
#include <termios.h>
#include <stdarg.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>

#include <libLPC2xxx.h>
#include <errcodes.h>
#include <posix_serial_port.h>

int serial_set_raw_mode( int fd )
{
  struct termios term;      /* attribute structure for setting port flags */

  /* fetch the existing port flags */
  if (tcgetattr(fd, &term) == -1)  return errno;

#ifdef linux
#if 0
  /* Set a variety of flags for raw mode.  See the tcsetattr man page. */
  term.c_iflag &= ~(BRKINT | PARMRK | INPCK | ISTRIP | INLCR | IGNCR | ICRNL | IUCLC | 
		    IXON | IXANY | IXOFF | IMAXBEL);
  term.c_iflag |= IGNBRK | IGNPAR;

  term.c_oflag &= ~(OPOST | OLCUC | ONLCR | OCRNL | ONOCR | ONLRET | OFILL | OFDEL | NLDLY |
		    CRDLY | TABDLY | BSDLY | VTDLY | FFDLY); 

  term.c_cflag &= ~(CSIZE | CSTOPB | PARENB | PARODD | HUPCL | CRTSCTS);
  term.c_cflag |=  CS8 | CREAD | CLOCAL;

  term.c_lflag &= ~(ISIG | ICANON | XCASE | ECHO | FLUSHO | TOSTOP | PENDIN | IEXTEN);  
  term.c_lflag |= NOFLSH;

#else
  cfmakeraw( &term );
#endif

  
#else
  /* Set a variety of flags for raw mode.  See the tcsetattr man page. */
  term.c_iflag &= ~(BRKINT | PARMRK | INPCK | ISTRIP | INLCR | IGNCR | ICRNL | IXON | IXANY | IXOFF | IMAXBEL);
  term.c_iflag |= IGNBRK | IGNPAR;

  term.c_oflag &= ~(OPOST | ONLCR | OXTABS | ONOEOT);

  term.c_cflag &= ~(CSIZE | CSTOPB | PARENB | PARODD | HUPCL | CCTS_OFLOW | CRTS_IFLOW | MDMBUF);
  term.c_cflag |=  CS8 | CREAD | CLOCAL;

  term.c_lflag &= ~(ECHOKE | ECHOE | ECHO | ECHONL | ECHOPRT | ECHOCTL | ISIG | ICANON | ALTWERASE | IEXTEN | EXTPROC 
		    | TOSTOP | FLUSHO | PENDIN);
  term.c_lflag |= NOFLSH | NOKERNINFO;
#endif

  // printf ("Setting descriptor %d to raw mode.\n", fd);
	  
  /* Set the new attributes immediately. */
  if (tcsetattr(fd, TCSANOW, &term) == -1) return errno;
  return 0;
}


int serial_set_baud_rate(int fd, int rate)
{
  struct termios term;      /* attribute structure for setting port flags */
  int baudsym;

  switch (rate){
  case 50: baudsym = B50; break;
  case 75: baudsym = B75; break;
  case 110: baudsym = B110; break;
  case 134: baudsym = B134; break;
  case 150: baudsym = B150; break;
  case 200: baudsym = B200; break;
  case 300: baudsym = B300; break;
  case 600: baudsym = B600; break;
  case 1200: baudsym = B1200; break;
  case 1800: baudsym = B1800; break;
  case 2400: baudsym = B2400; break;
  case 4800: baudsym = B4800; break;
  case 9600: baudsym = B9600; break;
  case 19200: baudsym = B19200; break;
  case 38400: baudsym = B38400; break;
  case 57600: baudsym = B57600; break;
  case 115200: baudsym = B115200; break;
  case 230400: baudsym = B230400; break;
  default:
    return -EINVAL;
  }

  /* fetch the existing port flags */
  if (tcgetattr(fd, &term) == -1) return -EBADF;

  // printf ("Setting baud rate of fd %d to %d\n", fd, rate);
  if (cfsetispeed(&term, baudsym) == -1) return errno;
  if (cfsetospeed(&term, baudsym) == -1) return errno;

  /* Set the new attributes immediately. */
  if (tcsetattr(fd, TCSANOW, &term) == -1) return errno;
  return 0;
}

static int convert_baudsym_to_rate( speed_t sym )
{
  switch (sym){
  case B50:    	 return ( 50     );
  case B75:    	 return ( 75     );
  case B110:   	 return ( 110    );
  case B134:   	 return ( 134    );
  case B150:   	 return ( 150    );
  case B200:   	 return ( 200    );
  case B300:   	 return ( 300    );
  case B600:   	 return ( 600    );
  case B1200:  	 return ( 1200   );
  case B1800:  	 return ( 1800   );
  case B2400:  	 return ( 2400   );
  case B4800:  	 return ( 4800   );
  case B9600:  	 return ( 9600   );
  case B19200:   return ( 19200  );
  case B38400:   return ( 38400  );
  case B57600:   return ( 57600  );
  case B115200:  return ( 115200 );
  case B230400:  return ( 230400 );
  default:
    return -EINVAL;
  }
}
int serial_get_input_baud_rate( int fd )
{
  struct termios term;      /* attribute structure for setting port flags */
  if (tcgetattr(fd, &term) == -1) return -EBADF;
  return convert_baudsym_to_rate ( cfgetispeed( &term ) );
}
int serial_get_output_baud_rate( int fd )
{
  struct termios term;      /* attribute structure for setting port flags */
  if (tcgetattr(fd, &term) == -1) return -EBADF;
  return convert_baudsym_to_rate ( cfgetospeed( &term ) );
}

/****************************************************************/
// Convenience routines for opening a port using libc I/O.

int
open_posix_serial_port_in_raw_mode( char *serial_port_name, int baud_rate, FILE **in, FILE **out)
{
  int input_fd;
  int output_fd;

  // clear any return variables
  if (in != NULL) *in = NULL;
  if (out != NULL) *out = NULL;

  // first open the port for input or output as needed as a primitive file,
  // then re-open the file as a character stream

  if ( in != NULL ) {
    if ((input_fd = open( serial_port_name, O_RDONLY, O_NOCTTY)) == -1) goto fail;
    if (serial_set_raw_mode( input_fd )) goto fail;
    if (serial_set_baud_rate( input_fd, baud_rate )) goto fail;
    if ((*in = fdopen( input_fd,  "r")) == NULL ) goto fail;
  }

  if ( out != NULL ) {
    if ((output_fd = open( serial_port_name, O_WRONLY)) == -1) goto fail;
    if (serial_set_baud_rate( output_fd, baud_rate )) goto fail;
    if ((*out = fdopen( output_fd,  "w")) == NULL ) goto fail;
  }
  return ERRNOERROR;
  
 fail:
  if (in != NULL) {
    if (*in != NULL) fclose( *in );
    else if ( input_fd >= 0) close( input_fd );
  }
  if (out != NULL) {
    if (*out != NULL) fclose(*out);
    else if ( output_fd >= 0) close( output_fd );
  }
  return -errno;
}
/*****************************************************************/
