// interrupt_control.c : utility routines to enable and disable interrupts
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------
#include <libLPC2xxx.h>
#include <interrupt_control.h>

void save_and_disable_interrupt_state ( UINT32 *state )
{
  // Disable FIQ and IRQ interrupts by setting the F and I bits in the
  // CPSR.  If the interrupts are not enabled than this doesn't affect
  // the CPSR.  Either way, the current state is saved so it may be
  // restored.

  // 1. copy the CPSR into tmp
  // 2. save the original CPSR value into *state
  // 3. set the interrupt mask bits to disable interrupts
  // 4. modify the CPSR with the new value

  // The & in the tmp constraint indicates that the register is
  // modified before the inputs are consumed; else the same register
  // might have been used for both input and output (tmp and state).

  register UINT32 tmp;

  asm volatile( "mrs %0, cpsr; str %0, [ %1 ]; orr %0, %0, #0xc0; msr cpsr_c, %0" 
		: "=&r" (tmp)
		: "r" (state)
		: "cc", "memory" );
}

void restore_interrupt_state ( UINT32 *state )
{
  register UINT32 tmp;

  // Load a CPSR value in order to restore the interrupt enable state.
  // The constraints specify an output value tmp to hold the value,
  // loaded indirectly using a register containing the argument
  // 'state'.  The third constraint specifies that the condition codes
  // are clobbered.  The volatile keyword is necessary since the
  // output value isn't used, and the optimizer would otherwise remove
  // the code as unnecessary.
  asm volatile( "ldr %0, [%1]; msr cpsr_c, %0"
		: "=r" (tmp)
		: "r" (state)
		: "cc" );
}

