// fifo.h : non-blocking FIFO implementation suitable for IPC between interrupt and normal contexts
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------
//! A basic non-blocking unlocked FIFO.  Rather than use malloc(), no
//! creation or destruction procedures are included, since these are
//! likely to be statically allocated.

#ifndef __FIFO_H_INCLUDED__
#define __FIFO_H_INCLUDED__

typedef struct fifo_t {
  void *buffer;
  int buffer_size;

  //! When data == space the buffer is empty.  The buffer is full when holding (buffer_size-1) bytes.
  volatile unsigned data;             // Position of first byte of data.
  volatile unsigned space;            // Position of first byte of empty space.
} fifo;

//! Safely initialize a fifo structure, either statically or dynamically allocated.
//! A pointer is saved to the supplied buffer, which is never directly freed.
extern fifo *init_fifo( fifo *p, void *buffer, int buffer_size  );

// Safely return the status of the fifo.
extern unsigned int fifo_space_available( fifo *p );
extern unsigned int fifo_data_available( fifo *p );
// extern unsigned int fifo_is_empty( fifo *p );
//extern unsigned int fifo_is_full( fifo *p );

// Atomic non-blocking write/read operations.
extern int fifo_try_put( fifo *p, void *data, unsigned int length );
extern int fifo_try_get( fifo *p, void *data, unsigned int length );
extern int fifo_try_read( fifo *p, void *data, unsigned int length );

extern int fifo_putchar( fifo *p, unsigned char byte);   // returns -ERRAGAIN if full, or zero on success.
extern int fifo_getchar( fifo *p );                      // returns EOF if nothing available

// Atomic peek operation.
extern int fifo_peek( fifo *p, void *data, unsigned int length );

#endif // __FIFO_H_INCLUDED__
