// chipinit.c : standard LPC2xxx CPU initialization
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------
#include <libstd.h>          // for memset()
#include <libLPC2xxx.h>
#include <interrupt_handlers.h>

// the main program entry point called by this file
extern int main(void);

// symbols defined by the linker to indicate memory sections
extern char _data_section_rom_start, _data_section_ram_start, _data_section_ram_end;
extern char _bss_section_start, _bss_section_end;

/****************************************************************/
// Configure a Pin Control Block entry.  This is most efficiently
// done by directly combining the masks in a single write in user
// code, but this is safe.
void PCB_configure_pin( int pin, int function)
{
  if ( pin < 16)       PCB.PINSEL0 = (PCB.PINSEL0 & ~PINSEL0_MASK(pin, 3)) | PINSEL0_MASK(pin, function);
  else if ( pin < 32)  PCB.PINSEL1 = (PCB.PINSEL1 & ~PINSEL1_MASK(pin, 3)) | PINSEL1_MASK(pin, function);
}

/****************************************************************/
void __main(void)
{
  // Initialize memory used by C programs.
  memset( &_bss_section_start, 0, &_bss_section_end - &_bss_section_start) ;
  memcpy( &_data_section_ram_start, &_data_section_rom_start, &_data_section_ram_end - &_data_section_ram_start );

#if CONFIG_CPU_LED_MASK
  // Initialize an I/O pin for the CPU board indicator LED, if there is one.
  CONFIG_CPU_LED_GPIO.IODIR |= CONFIG_CPU_LED_MASK;  // configure pin as output
  CONFIG_CPU_LED_GPIO.IOCLR  = CONFIG_CPU_LED_MASK;  // turn LED on
#endif

  // Initialize PLL to synthesize processor clock as follows:
  //   Fosc is is generated by the crystal.  Should be within 10-25 MHz.
  //   Fcco is the current-controlled PLL oscillator.  Should be within 156-320MHz.
  //   cclk is the PLL output and the processor clock.  Should be within 10-60Mhz.

  // This sets the following:
  //  SCB.PLLCFG.MSEL = CLOCKMULT-1  cclk is divided by msel+1 to compare against Fosc
  //  SCB.PLLCFG.PSEL = 01           P=2^PSEL (e.g. P=2), current-controlled clock is divided by 2P 
  //                                    e.g. 4) to generate CCLK
  SCB.PLLCFG  = 0x20 | (CLOCKMULT-1);

  SCB.PLLFEED = 0xAA;   // issue feed sequence for change to take effect
  SCB.PLLFEED = 0x55; 
  
  // Enabling the PLL.
  SCB.PLLCON   = PLLCON_PLLE_MASK;   // set PLLE, PLL enable
  SCB.PLLFEED = 0xAA;                // issue feed sequence for change to take effect
  SCB.PLLFEED = 0x55; 
  
  // Wait for the PLL to lock to set frequency.
  while( !(SCB.PLLSTAT & PLLSTAT_PLOCK_MASK ) );
  
  // Connect the PLL as the clock source.
  SCB.PLLCON  = PLLCON_PLLE_MASK | PLLCON_PLLC_MASK;  // PLLE = PLLC = 1, PLL enabled,  PLL connected
  SCB.PLLFEED = 0xAA;                                 // issue feed sequence for change to take effect
  SCB.PLLFEED = 0x55; 
  
  // Enabling the Memory Acceleration Module to cache FLASH memory reads.
  // The timing value used is that recommended for CPU clocks over 40 MHz;
  // if the processor clock speed is reduced, it could be reduced.
  MAM.TIM   = 0x03;  // MAM fetch cycles are 3 processor clocks (CCLK).
  MAM.CR    = 0x02;  // MAM functions fully enabled
  
  // Set peripheral clock (pclk) to the same frequency as the system
  // clock (cclk).  At reset, pclk starts at cclk/4.
  SCB.VPBDIV = 1;
  
  //-------- Initialize interrupt controller. -----------------
  VIC.IntSelect = 0x0;  // Assign all interrupt sources to IRQ, none to FIQ.

  // Set the default value returned on VIC.VectAddr if no slot has
  // been assigned to the source, which could either be a non-vectored
  // interrupt or a spurious interrupt.
  VIC.DefVectAddr = (unsigned long) default_IRQ_handler;

  // For each vectored interrupt, set the interrupt service routine
  // address, initialize a slot in the vectored interrupt response
  // table, and enable the interrupt.  Writes to IntEnable can only
  // set enabling bits, a separate register can clear them.

#if CONFIG_WATCHDOG_GENERATES_INTERRUPT
  VIC.VectAddr1   = (unsigned long) watchdog_interrupt_handler;
  VIC.VectCntl1 = VIC_VECTCNTL_ENABLED_MASK | VIC_WDT_CHANNEL;
  VIC.IntEnable = VIC_WDT_MASK;
#endif

  //----------- Initialize Timer 1 ---------------------

#if CONFIG_INSTALL_DEFAULT_TIMER_INTERRUPT
  // Set up the VIC for the interrupt
  VIC.VectAddr0   = (unsigned long) timer_interrupt_handler;
  VIC.VectCntl0 = VIC_VECTCNTL_ENABLED_MASK | VIC_TIMER1_CHANNEL;
  VIC.IntEnable = VIC_TIMER1_MASK;

  // Set up the timer
  TIMER1.TCR   = 0;    // turn it off
  TIMER1.TC    = 0;    // clear so we don't miss first match
  TIMER1.PR    = 0;    // prescale of 0 -> tick rate == sysclk
  TIMER1.PC    = 0;    // clear prescale counter for no particular reason

  // set the divisor for the timer interrupt
  TIMER1.MR0   = TIMER_INTERRUPT_DIVISOR;
  
  // Reset and interrupt on match to generate a periodic interrupt.
  TIMER1.MCR = TIMER_MCR_IMR0_MASK | TIMER_MCR_RMR0_MASK;  // RMR0 = IMR0 = 1

#endif // CONFIG_INSTALL_DEFAULT_TIMER_INTERRUPT

  //-----------  Initialize UART0 serial port. ------------------
#if CONFIG_UART0_DEFAULT_INIT
  // initialize UART
  UART0.FCR = 0x7;   // Enable and reset fifos (0x7); keep Rx Trigger Level at 0, to interrupt 
                     // after one character is received
  UART0.LCR = 0x83;  // 8 bits; enable divisor latches
  UART0.DLL = (UART0_DIVISOR & 0xff);  // LSB of divider for baud clock
  UART0.DLM = (UART0_DIVISOR >> 8);    // MSB of divider for baud clock
  UART0.LCR = 0x3;   // 8 bits; 1 stop bit; no parity; odd parity; disable break; disable divisor latches
  UART0.IER = 0;     // no interrupts enabled

  // enable UART0 in/out after the UART is configured
  PCB.PINSEL0 |= ( PINSEL0_MASK( 0, 1 ) |    // P0.0 is UART0 TxD 
		   PINSEL0_MASK( 1, 1 ) );   // P0.1 is UART0 RxD.  combined value is 5
#endif
  //================================================================

#if CONFIG_INSTALL_DEFAULT_TIMER_INTERRUPT
  // Start the timer interrupt clock.
  TIMER1.TCR = TIMER_TCR_ENABLE_MASK;
#endif

  //================================================================
  // Call the C main program
  main();
  
  // If it returns, stall.  If the watchdog it running, it will eventually trigger.
  while(1);
}
