// UART_raw_port.c : character device interface to polled serial ports
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------

// An abstract character device interface for polling UART0 and UART1
// with no buffers or interrupts.

#include <libstd.h>
#include <libLPC2xxx.h>
#include <UART_raw_port.h>
#include <port.h>
#include <errcodes.h>

static int write_char( struct port_t *port_ptr, int c)
{
  UART_raw_port *p = (UART_raw_port *) port_ptr;

  while (!(p->uart->LSR & LSR_THRE_MASK));  // wait for space in transmitter
  p->uart->THR = c;
  return ERRNOERROR;
}

static int char_ready( struct port_t *port_ptr )
{
  UART_raw_port *p = (UART_raw_port *) port_ptr;
  return ( p->uart->LSR & LSR_RDR_MASK);
}

static int read_char (struct port_t *port_ptr)
{
  UART_raw_port *p = (UART_raw_port *) port_ptr;

  while (!( p->uart->LSR & LSR_RDR_MASK));     // wait for character to be received
  return p->uart->RBR;
}

const static struct port_class_t UART_raw_class = {
  read_char,    // int  (*read_char)(struct port_t *p);
  NULL,         // int  (*peek_char)(struct port_t *p);
  char_ready,   // int  (*char_ready)(struct port_t *p);
  write_char,   // int  (*write_char)(struct port_t *p, int c);
  NULL, 	// size_t  (*read)(struct port_t *p, void *buf, size_t nbytes);
  NULL, 	// size_t (*write)(struct port_t *p, void *buf, size_t nbytes);
  NULL, 	// off_t   (*seek)(struct port_t *p, off_t offset, int whence);
  NULL  	// int    (*close)(struct port_t *p);
};

/****************************************************************/
int 
UART_raw_port_init( UART_raw_port *p, struct UART_registers_t *uart, unsigned divisor, unsigned mode )
{
  generic_port_init( &p->interface, &UART_raw_class, mode );
  p->uart = uart;

  // Initialize the hardware port.
  // initialize UART
  uart->FCR = 0x7;   // Enable and reset fifos (0x7); keep Rx Trigger Level at 0, to interrupt 
                     // after one character is received
  uart->LCR = 0x83;  // 8 bits; enable divisor latches
  uart->DLL = (divisor & 0xff);  // LSB of divider for baud clock
  uart->DLM = (divisor >> 8);    // MSB of divider for baud clock
  uart->LCR = 0x3;   // 8 bits; 1 stop bit; no parity; odd parity; disable break; disable divisor latches
  uart->IER = 0;     // no interrupts enabled

  // enable UART pins after the UART is configured
  if ( uart == &UART0 ) {
    if ( mode & PORT_FLAGS_READ )  PCB.PINSEL0 |= PINSEL0_MASK( 0, 1 );  // P0.0 is UART0 TxD 
    if ( mode & PORT_FLAGS_WRITE ) PCB.PINSEL0 |= PINSEL0_MASK( 1, 1 );  // P0.1 is UART0 RxD.
  } else if ( uart == &UART1 ) {
    if ( mode & PORT_FLAGS_READ )  PCB.PINSEL0 |= PINSEL0_MASK( 8, 1 );  // P0.8 is UART1 TxD 
    if ( mode & PORT_FLAGS_WRITE ) PCB.PINSEL0 |= PINSEL0_MASK( 9, 1 );  // P0.9 is UART1 RxD.
  }
  return ERRNOERROR;
}
