// sensors.c : convert raw sensor readings into calibrated values
// This file is part of ArtLPC. 
// This file is placed into the public domain.

// This should reflect the actual configuration of the robot.

#include "sensors.h"

/****************************************************************/
// inline function for applying offset and scale to an integer sensor value
inline float calibrate( int value, int imin, int imax, float fmin, float fmax ) 
{
  return fmin + (((float)(value - imin)) * ((fmax - fmin) / (float) (imax-imin)));
}

#define WHEEL_DIAMETER (0.104)

void compute_calibrated_sensor_values(void)
{
  // The wheels are nominally 104 mm diameter, with 4000 total counts
  // per revolution.  The calibrated value is in meters.
  q.left  = (float) q_raw.left  * (M_PI*WHEEL_DIAMETER / 4000);
  q.right = (float) q_raw.right * (M_PI*WHEEL_DIAMETER / 4000);

  // Calibrate the accelerometer values based on measured inputs.
  // This uses the vertical orientation as one reference since the
  // machine is most likely to operate near this point.
  qdd.x = calibrate( qdd_raw.x,   -1, 103,  0.00, 9.81 );
  qdd.y = calibrate( qdd_raw.y,   -8,  96,  0.00, 9.81 );
  qdd.z = calibrate( qdd_raw.z,   -1, 107,  0.00, 9.81 );

  // The IDG-300 rate gyro has a nominal +/- 500 degree/second range.
  // The nominal sensitivity is 2mv / deg/sec, which implies a total output
  // range of 2V.  The LP2103 A/D converter maps a [0,3.3V] input to a
  // [0, 1023] code range.  So, after zero-centering the raw value, this 
  // calibrates the [-1V, 1V] range to the [-500, 500] radians/second.

#define RADIANS(deg) (M_PI*(deg)/180)
#define ONEVOLTCODE ((int)(1024 / 3.3))

  qd.pitch = calibrate( qd_raw.pitch - 43, -ONEVOLTCODE, ONEVOLTCODE, RADIANS(-500), RADIANS(500) );
  qd.yaw   = calibrate( qd_raw.yaw   + 38, -ONEVOLTCODE, ONEVOLTCODE, RADIANS(-500), RADIANS(500) );

  // ground angle encoder
  q.tilt = calibrate( q_raw.tilt, 0, 4096, 0.0, M_PI ); // 2048 CPR==8192 discrete positions

}
/****************************************************************/
void estimate_velocities(void)
{
  static struct q_t q_last;

#define SAMPLE_RATE (100.0)  // defined in the Balancer code
  // use the simplest possible forward differencing to estimate velocity
  qd.left  = (q.left  - q_last.left ) * SAMPLE_RATE;
  qd.right = (q.right - q_last.right) * SAMPLE_RATE;

  // simple differencing of the ground sensor encoder, almost the same
  // as pitch velocity
  qd.tilt = (q.tilt - q_last.tilt) * SAMPLE_RATE;

  // save the current value for the next cycle
  q_last = q;
}
#undef SAMPLE_RATE

/****************************************************************/
