// sabertooth.c : interface to a Dimension Engineering Sabertooth 2x10 motor amplifier
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------

#include <libLPC2xxx.h>
#include <libstd.h>
#include <errcodes.h>
#include <timing.h>
#include <UART_port.h>

#include <sabertooth.h>

// The Sabertooth documentation recommends a two second delay after
// powerup before sending any commands to the amplifier.
#define SABERTOOTH_STARTUP_DELAY MSEC_TO_TICKS(2000)

/****************************************************************/
// globals
struct sabertooth_t sabertooth;

static UART_port amplifier_UART_port;
static struct port_t *amplifier_port = NULL;

/****************************************************************/
int
init_sabertooth(void)
{
  // open UART1 for the motor amplifier, which is output only
  UART_port_init( &amplifier_UART_port, &UART1, UART1_DIVISOR, PORT_FLAGS_WRITE );
  amplifier_port = (struct port_t *) &amplifier_UART_port;

  // enforce a startup delay on any commands
  sabertooth.initialized = 0;
  elapsed_timer_init( &sabertooth.timer );

  // provide an indicator of whether commands are generally being issued
  sabertooth.active = 0;
  return ERRNOERROR;
}

/****************************************************************/
// Issue a basic motor command in the native units.
int
sabertooth_command( int channel, int value )
{
  int command, data;
  unsigned checksum;

  // if the startup delay hasn't been satisfied, don't send anything
  if (!sabertooth.initialized) return -ERRAGAIN;

  // else reset the command timer after each command is issued to
  // avoid our shutdown timeout
  elapsed_timer_init( &sabertooth.timer );

  // and flag that commands are streaming
  sabertooth.active = 1;

  if ( channel == 0 ) {
    if (value >= 0) {
      command = 0;  // motor 1 forward
      data = value;
    } else {
      command = 1;  // motor 1 backward
      data = -value;
    }
  } else {
    if (value >= 0) {
      command = 4;  // motor 2 forward
      data = value;
    } else {
      command = 5;  // motor 2 backward
      data = -value;
    }
  }
  // limit the maximum command
  if (data > 127) data = 127;

  // compute the checksum
#define AMPLIFIER_ADDRESS 128
  checksum = (AMPLIFIER_ADDRESS + command + data) & 0x7f;

  port_write_char( amplifier_port, 0xaa );               // send the bauding character
  port_write_char( amplifier_port, AMPLIFIER_ADDRESS );
  port_write_char( amplifier_port, command);
  port_write_char( amplifier_port, data);
  port_write_char( amplifier_port, checksum);

  return ERRNOERROR;
}
/****************************************************************/
// This function should be called periodically from the main event
// loop to manage any internal state.

int sabertooth_update(void)
{

  // if the start up delay is still in effect, wait for it to
  // complete, and then flag readiness
  if ( !sabertooth.initialized ) {
    if (elapsed_timer_update( &sabertooth.timer ) > SABERTOOTH_STARTUP_DELAY ) {
      sabertooth.initialized = 1;
      sabertooth.active = 0;
      elapsed_timer_init( &sabertooth.timer );
    }
  } 
  // If running normally, check if too long an interval has passed
  // between output commands, and if so, then shut the motor down for
  // safety.  Issuing the command will also reset the driver; the
  // driver will thus periodically get this command when no host is
  // attached.
  else if (elapsed_timer_update( &sabertooth.timer ) > MSEC_TO_TICKS( 1000 )) {
    sabertooth_command( 0, 0 );
    sabertooth_command( 1, 0 );
    sabertooth.active = 0;
    format( NULL, "# Sabertooth 2x10 command timeout, issuing motor off sequence.\n");
  }

  return ERRNOERROR;
}
/****************************************************************/
