// main.c : top-level source file for the Balancer robot controller
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------

#include <libstd.h>
#include <string_port.h>
#include <errcodes.h>
#include <libLPC2xxx.h>
#include <LED_indicators.h>
#include <UART_port.h>
#include <timing.h>
#include <interrupt_handlers.h>

#include <timer_interrupt.h>
#include <encoder_counter.h>
#include <imu.h>
#include <sabertooth.h>

/****************************************************************/
// A simple demo controller.
static int demo_active = 0;

static void compute_control(void)
{
  // simple proportional control to keep wheels at zero
  int tau1, tau2;

  tau1 = ( 0 - -encoder1.position) >> 6;
  tau2 = ( 0 -  encoder2.position) >> 6;

  sabertooth_command( 0, -tau1 );
  sabertooth_command( 1, -tau2 );
}
static void motor_power_off(void)
{
  sabertooth_command( 0, 0 );
  sabertooth_command( 1, 0 );
}

/****************************************************************/
// provide a resettable timer to control the data indicator LED;
// as long as this is below some value, light the lamp
#define INDICATOR_PULSE_WIDTH MSEC_TO_TICKS(50)
static elapsed_timer data_indicator_timer;

/****************************************************************/
// Read an ASCII command line incrementally without blocking, and then
// parse it when complete.

static void process_control_stream(void)
{
#define MAX_COMMAND_LENGTH 40
  static char linebuf[MAX_COMMAND_LENGTH+1];
  static int next_char = 0;
  
  if (port_char_ready(current_input_port)) {
    int c = port_read_char(current_input_port);

    elapsed_timer_init( &data_indicator_timer );

    if ( c != '\n' ) {
      // just store data until the line is complete
      if ( next_char < MAX_COMMAND_LENGTH ) linebuf[next_char++] = c;      
      return;

    } else {
      // if line is done, parse a complete command
      char *cptr = linebuf;
      int uleft = 0, uright = 0;

      // terminate the buffer and reset the pointer
      linebuf[next_char] = 0;
      next_char = 0;

      // process each element of the line
      for(;;) {
	// scan for a command character
	while (*cptr && !isalpha(*cptr)) cptr++;

	switch(*cptr) {

	case 'L':
	  uleft = atoi(cptr+1);
	  if (!demo_active) sabertooth_command( 0, -uleft );
	  break;

	case 'R':
	  uright = atoi(cptr+1);
	  if (!demo_active) sabertooth_command( 1, -uright );
	  break;
	  
	case 0: // line is done
	  // format( NULL, "# uleft ~d uright ~d\n", uleft, uright );
	  return;
	}
	// advance past the command char, which will then scan through
	// the number until it finds another command char or the end
	cptr++; 
      }
    }
  }
}
/****************************************************************/
int main (void)
{
  static UART_port console_port;
  elapsed_timer sleep_timer;
  unsigned last_buttons = 0;

  // set up the hardware as soon as possible
  init_encoder_counter();
  init_imu();
  init_sabertooth();
  init_timer_interrupt();
  LED_indicators_init();
  elapsed_timer_init( &data_indicator_timer );

  // open the console port and issue a welcome message
  UART_port_init( &console_port, &UART0, UART0_DIVISOR, PORT_FLAGS_READ|PORT_FLAGS_WRITE|PORT_FLAGS_CRLF );
  current_output_port = (struct port_t *) &console_port;
  current_input_port  = (struct port_t *) &console_port;

  format( NULL, 
	  "# Balancer Controller.\n"
	  "# part of ArtLPC, Copyright (C) 2005-2007 Garth Zeglin\n"
	  "# This is free software; see the source for copying conditions.  There is NO\n"
	  "# warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n"
	  "# Compiled " __DATE__ " " __TIME__ " from " __FILE__ "\n" 
	  "# Protocol version 1\n"
	  );

  // debug_imu();

  for (;;) {
    unsigned buttons;

    CPU_LED_on();    // start the event loop

    // On every cycle, emit a sensor value record.  The sign bits are
    // corrected as needed for the coordinate system. Only the integer
    // portion of each fixed point IMU value is reported.
    format( current_output_port, "L~dR~dX~dY~dZ~dP~dQ~dT~d\n",
	    -encoder1.position, encoder2.position,
	    imu.value[2]  >> IMU_MANTISSA,  // xaccel
	    -imu.value[1] >> IMU_MANTISSA,  // yaccel
	    imu.value[0]  >> IMU_MANTISSA,  // zaccel
	    -imu.value[4] >> IMU_MANTISSA,  // pitchvel
	    imu.value[3]  >> IMU_MANTISSA,  // yawvel
	    extra_encoder.position          // encoder on expansion connector
	    );
   
    // the "user interface"
    buttons = (GPIO0.IOPIN & CONFIG_BUTTON1_MASK) ? 0 : 1; // switch is active-low
    if ( buttons ^ last_buttons ) {
      if (buttons & 1) {
	format( NULL, "# button pressed, engaging demo controller\n");
	demo_active = 1;
      } else {
	format( NULL, "# button released, disengaging demo controller\n");
	demo_active = 0;
	motor_power_off();
      }
    }
    last_buttons = buttons;

    // provide something to do for testing
    if (demo_active) compute_control();

    // poll the motor amplifier driver
    sabertooth_update();

    // generate a status LED update
    LED_indicators_set( ((sabertooth.initialized) ?  LED_YEL1 : LED_RED1) |
			((sabertooth.active) ? LED_GRN1 : 0) |
			((elapsed_timer_update(&data_indicator_timer) < INDICATOR_PULSE_WIDTH) ? LED_GRN2 : 0) |
			((timerclock & 32768) ? LED_YEL2 : 0) |  // busybox
			((demo_active) ? LED_RED2 : 0));
      
    CPU_LED_off();

    // kill some time to slow down the main cycle to 100Hz, but still process input data.
    elapsed_timer_init( &sleep_timer );
    while ( elapsed_timer_update( &sleep_timer ) < MSEC_TO_TICKS( 10 )) {
      process_control_stream();      // process any control command data
    }
  }
}


