// lqr-test.c : run a Balancer controller on a host using a serial port for I/O
// This file is part of ArtLPC. 
// This file is placed into the public domain.

// The common bits of host-side controller code are separated into a
// couple of other files.  For compilation simplicity, those files are
// simply included directly in this one.

#include <stdio.h>
#include <math.h>

// handle for a file in which to save data
static FILE *data_trace_out;

// overall time counter
static unsigned cycle_count = 0;

#include "matrix.c"

/****************************************************************/
// Set up the particular linear control model.

#define DT (0.01)  // integration time constant for model update

#define ACCEL_GYRO_MODEL 1
#define ACCEL_ONLY_MODEL 2
#define POSITION_SENSOR_MODEL 3
#define MODEL 3

#if MODEL==ACCEL_GYRO_MODEL
#include "linear-control.c"  // define the linear control state and functions
#endif

#if MODEL==ACCEL_ONLY_MODEL
#include "linear-control-no-gyro.c"  // define the linear control state and functions
#endif

#if MODEL==POSITION_SENSOR_MODEL
#include "linear-control-pitch-encoder.c"  // define the linear control state and functions
#endif

/****************************************************************/
// Include the sensor calibration code.
#include "sensors.c"

/****************************************************************/
// Include the robot interface.
#include "serial-robot-interface.c"

/****************************************************************/
static void display_data(void)
{
  
#if 1
  // display calibrated data to user
  printf( "q.left: %f q.right: %f  q.tilt: %f Xe:[ %f %f %f %f ] --> U: %f\n",
	  q.left, q.right, q.tilt,
	  modelXe[0],modelXe[1],modelXe[2],modelXe[3],
	  modelU[0] );
#endif
}

/****************************************************************/
static void log_data(void)
{
  // save all raw data in a file
  if ( data_trace_out != NULL ) {
    fprintf( data_trace_out, "%d %d %d %d %d %d %d %d %d %d %d\n",
	     cycle_count, 
	     q_raw.left, q_raw.right, q_raw.tilt,
	     qdd_raw.x, qdd_raw.y, qdd_raw.z, 
	     qd_raw.pitch, qd_raw.yaw, 
	     tau_raw.left, tau_raw.right );
  }
}
/****************************************************************/
int main (int argc, char **argv)
{
  int err;

  printf("opening serial port to communicate with robot.\n");
  err = open_posix_serial_port_in_raw_mode( SERIAL_PORT, 115200, &serial_sensor_in, &serial_command_out );

  if (err) {
    printf("failed to open serial port, error %d\n", err);
    exit(1);
  }

  data_trace_out = fopen( "trace.data", "w");

  // read sensor data and emit controls
  for( ; ; cycle_count++) {

    // wait for a sensor packet from the robot, quitting on any error
    if ( wait_for_sensor_data() ) goto done;

    // convert the integer sensor values to something meaningful
    compute_calibrated_sensor_values();
    estimate_velocities();

    // Rather than use a Kalman filter, this model has sensors which
    // directly measure the desired position state, and uses simple
    // differencing calculations to compute velocity, and LQR to
    // generate control outputs.

    modelXe[0] = 0.5*(q.left + q.right);
    modelXe[1] = 0.5*(qd.left + qd.right);
    modelXe[2] = q.tilt;
    modelXe[3] = qd.tilt;

    // compute a LQR to balancer around vertical (the zero state vector)
    update_LQR_control();      // compute the modelU control vector based on the state estimate

    // just reduce the torques a lot for testing
    modelU[0] *= 0.4;

    if (modelU[0] > 0.75) modelU[0] = 0.75;
    else if (modelU[0] < -0.75) modelU[0] = -0.75;

    tau.left  = modelU[0];
    tau.right = modelU[0];

    // In the future, this should include an additional term to keep
    // the wheels rotating the same amount.

    // update the motor state
    send_commanded_torque_to_robot();

    // debugging output
    display_data();
    log_data();

  }

 done:
  printf("control loop exited.\n");
  fclose( serial_sensor_in );
  fclose( serial_command_out );
  if ( data_trace_out != NULL ) fclose ( data_trace_out );
  return 0;
}
/****************************************************************/


