// linear-control-pitch-encoder.c : Kalman filter, LQR, and linear simulator for the Balancer robot
// This file is part of ArtLPC. 
// This file is placed into the public domain.

/****************************************************************/
// The control is linearized around the vertical (the zero state vector).

// The state vector is defined as [ x ; x' ; angle ; angle' ], where x
// is the linear position along the ground, and angle is the pitch of
// the body w.r.t. gravity.

// The input vector has the motor torque [ u ].

// The output vector has two position sensors [ wheelanglesensor bodypitchsensor ]

/****************************************************************/
#define NSTATES   4   // internal states of the linear system
#define NOUTPUTS  2   // observable sensor outputs from the linear system
#define NCONTROLS 1   // number of control inputs to the linear system

// Model matrices for the Kalman filter and LQR. These are stored
// C-style in row-major order.

static float modelA[NSTATES*NSTATES] = {
  0,  1,           0,           0 ,
  0,  0,   -5.701513732355099,  0 ,
  0,  0,           0,           1 ,
  0,  0,   22.2715380170121,    0 ,
};

static float modelB[NSTATES*NCONTROLS] = {
  0 , 
  15.56284804906838 , 
  0 , 
  -25.02085504515322, 
};

static float modelC[NOUTPUTS*NSTATES] = {
  19.2308,    0.0,      -1.0,    0.0,      // wheel encoder in radians
  0.0,        0.0,       1.0,    0.0,      // body pitch measurement
};

static float modelNegC[NOUTPUTS*NSTATES] = {
  -19.2308,   0.0,       1.0,     0.0,
    0.0,      0.0,      -1.0,     0.0,
};

static float modelD[NOUTPUTS*NCONTROLS] = {
  0.0,
  0.0 
};

static float modelNegD[NOUTPUTS*NCONTROLS] = {
  0.0,  // the x accelerometer is coupled to the torque
  0.0 
};

// gain matrix for the Kalman observer
static float kalmanL[NSTATES*NOUTPUTS] = {
  0.662272,    0.022634,
  6.896659,   -2.674373,
  -4.387924,    4.823194,
  -23.613546,  21.258540
};

// gain matrix for the LQR controller
static float lqrNegK[NCONTROLS*NSTATES] = {
  1.0000,  1.7059,  8.6731,  2.4858
};

/****************************************************************/
// Output (sensor) vector for the linear controller; this is where the
// current sensor readings are supplied.
static float modelY[ NOUTPUTS ];

// State estimate for the linear controller.  This represents the best
// guess as to the actual system state, and is used for control.
static float modelXe[ NSTATES ];

// Control vector for the linear controller.  This is the control
// computed to be applied to the actual system.
static float modelU[ NCONTROLS ];

// State vector for the linear simulation.  This is used when testing
// the controller on a block linear simulation.
static float modelX[ NSTATES ];

/****************************************************************/
// Recompute the state estimate based on the sensor readings, the
// previous state estimate, the physics model, and the Kalman L gain
// matrix.

void update_Kalman_filter(void)
{
  int i;
  float dXedt[ NSTATES ];
  float Yerror[ NOUTPUTS ];

  // compute the state estimator update based on the output and
  // previous state estimate from time n-1:
  //   dXe/dt = A*xe + B*u + L * ( y - C*xe - D*u );

  // dXedt = A*xe + B*u 
  matrix_zero( dXedt, NSTATES*1 );
  matrix_multiply_sum( modelA, modelXe, dXedt, NSTATES, NSTATES, 1 );
  matrix_multiply_sum( modelB, modelU,  dXedt, NSTATES, NCONTROLS, 1 );
  
  // Yerror = ( y - C*xe - D*u )
  matrix_copy( modelY, Yerror, NOUTPUTS*1 );
  matrix_multiply_sum( modelNegC, modelXe, Yerror, NOUTPUTS, NSTATES, 1 );
  matrix_multiply_sum( modelNegD, modelU,  Yerror, NOUTPUTS, NCONTROLS, 1 );

  // putting it all together: dXedt = dXedt + L * ( Yerror );
  matrix_multiply_sum( kalmanL, Yerror, dXedt, NSTATES, NOUTPUTS, 1);

  // apply forward Euler integration into the state estimate
  for ( i = 0; i < NSTATES; i++ ) modelXe[i] += DT * dXedt[i];

}
/****************************************************************/
// Recompute the control output based on the estimated state.
void update_LQR_control(void)
{
  // u = -K * xe;
  matrix_zero( modelU, NCONTROLS );
  matrix_multiply_sum( lqrNegK, modelXe, modelU, NCONTROLS, NSTATES, 1 );
}

/****************************************************************/
// Simulate the system using the linearized equations.  This can be
// used for desk-checking the LQR and Kalman gains; if it fails with
// this, there is something broken.

void update_linear_simulation(void)
{
  int i;
  float dXdt[ NSTATES ];

  // y = C*x + D*u
  matrix_zero( modelY, NOUTPUTS*1 );
  matrix_multiply_sum( modelC, modelX, modelY, NOUTPUTS, NSTATES, 1 );
  matrix_multiply_sum( modelD, modelU, modelY, NOUTPUTS, NCONTROLS, 1 );

  // dxdt = A*x + B*u
  matrix_zero( dXdt, NSTATES*1 );
  matrix_multiply_sum( modelA, modelX,  dXdt, NSTATES, NSTATES, 1 );
  matrix_multiply_sum( modelB, modelU,  dXdt, NSTATES, NCONTROLS, 1 );

  // apply forward Euler integration into the simulated state
  // x += dxdt * dt
  for ( i = 0; i < NSTATES; i++ ) modelX[i] += DT * dXdt[i];

}  
/****************************************************************/  

