// imu.c : interface to a Spark Fun 5DOF analog IMU
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------

#include <libLPC2xxx.h>
#include <libstd.h>
#include <errcodes.h>
#include <timing.h>
#include <imu.h>

// globals
struct imu_t imu;

/****************************************************************/
//  PCB.PINSEL0 == 0x00000305
//  PCB.PINSEL1 == 0x05500000


int
init_imu(void)
{
  int i;
  // configure the five analog signal lines as A/D inputs
  PCB_configure_pin( 22, 3 );   // P0.22 as AD0.0
  PCB_configure_pin( 23, 3 );   // P0.23 as AD0.1
  PCB_configure_pin( 24, 3 );   // P0.24 as AD0.2
  PCB_configure_pin( 12, 3 );   // P0.12 as AD0.5
  PCB_configure_pin( 25, 3 );   // P0.25 as AD0.6
  
  for (i = 0; i < 5; i++) {
    imu.value[i]     = 0 << IMU_MANTISSA; // clear the initial values to midscale
    imu.zeropoint[i] = 0 << IMU_MANTISSA; // initialize the DC offsets to be approximately correct
  }
  imu.samples = 0;
  return ERRNOERROR;
}

/****************************************************************/
void debug_imu(void)
{
  format( NULL, "debug_imu:\n" );
  format( NULL, "  ADC_CLOCK_DIVISOR == ~d\n", ADC_CLOCK_DIVISOR );
  format( NULL, "  value[0] == ~d\n", imu.value[0]);
  format( NULL, "  PCB.PINSEL0 == 0x~x\n", PCB.PINSEL0 );
  format( NULL, "  PCB.PINSEL1 == 0x~x\n", PCB.PINSEL1 );
}

/****************************************************************/
// This is called from interrupt context. It sequentially triggers the
// A/D converter for each IMU channel at a rate which is a fixed
// fraction of the interrupt rate.

void imu_poll(void)
{
  // define the hardware channels on ADC0 corresponding to the IMU channels
  const static unsigned AD_channel[5] = { 0, 1, 2, 5, 6 };

  unsigned channel, sequence;

  sequence   = timerclock & 1;
  channel    = (timerclock >> 1) & 7;

  // ignore channel numbers not in use by the IMU
  if ( channel < 5) {
    switch (sequence) {

      // when the low order bits are 0, trigger the A/D for the next reading
    case 0:
      ADC0.ADCR = ( ( 1 << AD_channel[channel] ) |     // channel select bitfield
		    ( ADC_CLOCK_DIVISOR << 8 ) |       // A/D clock = PCLK / (CLKDIV+1); this rounds down by 
		    0 |                                // BURST = 0
		    0 |                                // CLKS  = 0
		    ADCR_PDN_MASK |                    // enable converter
		    ( 1 << 24 ) |                      // start now
		    0 );                               // EDGE = 0
      break;
      
    case 1: // on the next interrupt, read the value and process it
      {
	fixed_point_t error, feedback;

	// Read the sample, get the zero offset approximately
	// correct, and convert it to a fixed point format to increase
	// the precision of the filtering.

	fixed_point_t sample = (((ADC0.ADDR & ADDR_DATA_MASK ) >> (ADDR_DATA_SHIFT)) - 512) << IMU_MANTISSA;

	// save the raw data
	imu.raw_sample[channel] = sample;
	imu.samples++;
      
	// apply a simple smoothing filter to remove conversion noise
	imu.value[channel] += ((sample - imu.value[channel]) / 4);

	// apply a long-term averaging filter to get a DC bias
	error = imu.value[channel] - imu.zeropoint[channel];
	feedback = error >> 15;
	if ( feedback == 0 ) {  // protect against truncation underflow
	  if ( error < 0 )      feedback = -1;
	  else if ( error > 0 ) feedback =  1;
	}
	imu.zeropoint[channel] += feedback;
      }
      break;
    }
  }
}  

/****************************************************************/
