// example-controller.c : run a Balancer controller on a host using a serial port for I/O
// This file is part of ArtLPC. 
// This file is placed into the public domain.

// The common bits of host-side controller code are separated into a
// couple of other files.  For compilation simplicity, those files are
// simply included directly in this one.

#include <stdio.h>
#include <math.h>

static FILE *serial_sensor_in, *serial_command_out;
static FILE *data_trace_out;

/****************************************************************/
// Include the sensor calibration code.
#include "sensors.c"

/****************************************************************/
// Include the robot interface.
#include "serial-robot-interface.c"

/****************************************************************/
// overall time counter
static unsigned cycle_count = 0;

/****************************************************************/
// Provide some numbers for the user to see.
static void display_data(void)
{
  
#if 0
  // display uncalibrated data
  printf( "q.left: %6d q.right: %6d  qdd.x: %6d qdd.y: %6d qdd.z: %6d  qd.pitch: %6d qd.yaw %6d\n",
	  q_raw.left, q_raw.right, 
	  qdd_raw.x, qdd_raw.y, qdd_raw.z, 
	  qd_raw.pitch, qd_raw.yaw );
#endif

#if 1
  // display calibrated data to user
  printf( "q.l: %f q.r: %f q.t: %f qdd.x: %f qdd.y: %f qdd.z: %f  qd.pitch: %f qd.yaw %f tau.l: %f tau.r: %f\n",
	  q.left, q.right, q.tilt,
	  qdd.x, qdd.y, qdd.z, 
	  qd.pitch, qd.yaw, 
	  tau.left, tau.right );
#endif

#if 0
  // display calibrated data to user
  printf( "q.left: %f q.right: %f  qd.l: %f qd.r %f  qdd.x: %f qdd.y: %f qdd.z: %f  --> tau.l: %f tau.r: %f\n",
	  q.left, q.right, 
	  qd.left, qd.right, 
	  qdd.x, qdd.y, qdd.z, 
	  tau.left, tau.right );
#endif

}

/****************************************************************/
// Dump all the raw state and output data to a file for post-processing.
static void log_data(void)
{
  // save all raw data in a file
  if ( data_trace_out != NULL ) {
    fprintf( data_trace_out, "%d %d %d %d %d %d %d %d %d %d %d\n",
	     cycle_count, 
	     q_raw.left, q_raw.right, q_raw.tilt,
	     qdd_raw.x, qdd_raw.y, qdd_raw.z, 
	     qd_raw.pitch, qd_raw.yaw, 
	     tau_raw.left, tau_raw.right );
  }
}
/****************************************************************/
int main (int argc, char **argv)
{
  int err;

  printf("opening serial port to communicate with robot.\n");
  err = open_posix_serial_port_in_raw_mode( SERIAL_PORT, 115200, &serial_sensor_in, &serial_command_out );

  if (err) {
    printf("failed to open serial port, error %d\n", err);
    exit(1);
  }

  data_trace_out = fopen( "trace.data", "w");

  // read sensor data and emit controls
  for( ; ; cycle_count++) {

    // wait for a sensor packet from the robot, quitting on any error
    if ( wait_for_sensor_data() ) goto done;

    // convert the integer sensor values to something meaningful
    compute_calibrated_sensor_values();
    estimate_velocities();

    // compute some trivial control function
#if 1
    // Compute a trivial position servo using the wheel encoders, just
    // to make sure the system is functioning.
    tau.left  = (0.5 * (0.0 - q.left )) - (0.01 * qd.left );
    tau.right = (0.5 * (0.0 - q.right)) - (0.01 * qd.right);
#endif

#if 0
    tau.left  = (0.5 * (0.0 - q.left )) - (0.01 * qd.left ) - (0.1 * qdd.x);
    tau.right = (0.5 * (0.0 - q.right)) - (0.01 * qd.right) - (0.1 * qdd.x);
#endif

#if 0
    tau.left  = (-0.1 * qdd.x) + (0.5 * qd.pitch);
    tau.right = tau.left;
#endif

#if 0
    tau.left  = (2.0 * q.tilt);
    tau.right = tau.left;
#endif

    // update the motor state
    send_commanded_torque_to_robot();

    // debugging output
    display_data();
    log_data();

  }

 done:
  printf("control loop exited.\n");
  fclose( serial_sensor_in );
  fclose( serial_command_out );
  if ( data_trace_out != NULL ) fclose ( data_trace_out );
  return 0;
}
/****************************************************************/


