// encoder_counter.c : two channel quadrature encoder interface
// Copyright (c) 2005-2007 Garth Zeglin

// This file is part of ArtLPC. 

// ArtLPC is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// ArtLPC is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with ArtLPC; if not, write to the Free Software Foundation,
// Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

// ---------------------------------------------------------------------

#include <libLPC2xxx.h>
#include <libstd.h>
#include <errcodes.h>
#include <encoder_counter.h>

// globals
struct encoder_t encoder1, encoder2;

#if CONFIG_ENCODER_ON_EXPANSION
struct encoder_t extra_encoder;
#endif

/****************************************************************/
int
init_encoder_counter(void)
{
  // the GPIO lines are configured as inputs by default, so nothing needs
  // to happen with the hardware

  encoder1.position = 0;
  encoder1.last_step = 0;
  encoder1.last_phase = 0;
  encoder1.A_mask = MOT1APIN;
  encoder1.B_mask = MOT1BPIN;
  encoder1.I_mask = MOT1IPIN;
  encoder1.ABI_mask = MOT1APIN | MOT1BPIN | MOT1IPIN; // precompute this for speed

  encoder2.position = 0;
  encoder2.last_step = 0;
  encoder2.last_phase = 0;
  encoder2.A_mask = MOT2APIN;
  encoder2.B_mask = MOT2BPIN;
  encoder2.I_mask = MOT2IPIN;
  encoder2.ABI_mask = MOT2APIN | MOT2BPIN | MOT2IPIN; // precompute this for speed

#if CONFIG_ENCODER_ON_EXPANSION
  extra_encoder.position = 0;
  extra_encoder.last_step = 0;
  extra_encoder.last_phase = 0;
  extra_encoder.A_mask = EXTRAAPIN;
  extra_encoder.B_mask = EXTRABPIN;
  extra_encoder.I_mask = EXTRAIPIN;
  extra_encoder.ABI_mask = EXTRAAPIN | EXTRABPIN | EXTRAIPIN; // precompute this for speed
#endif
  
  return ERRNOERROR;
}

/****************************************************************/
// This is called from interrupt context. It samples the quadrature
// encoder state for a single encoder and keeps track of the overall
// position.

static void poll_encoder( struct encoder_t *dev )
{
  // The values encode the quadrature relationship.  A 2 is ambiguous,
  // and can indicate a double step in the current direction.
  static const int phase_difference_table[4][4] = 
    { {  0, -1,  1,  2 },  
      {  1,  0,  2, -1 },
      { -1,  2,  0,  1 },
      {  2,  1, -1,  0 } };

  unsigned quad_bits, phase;
  int step;

  // Read the quadrature lines and convert to a phase value.  The bits
  // are synchronously sampled, then masked off to yield a phase code which
  // goes in sequence: 0 1 3 2 0 1 3 2 ...
  quad_bits  = GPIO0.IOPIN & dev->ABI_mask;
  phase      = ((quad_bits & dev->A_mask) ? 1 : 0) + ((quad_bits  & dev->B_mask) ? 2 : 0);

  // The phase code value is an integer on [0, 3].  If moving quickly, the
  // motor could move multiple phases between polling; take the best
  // guess at the difference by assuming it is on the range [-1, 1].
  // If the difference is 2 or -2, then assume it is still moving in
  // the same direction.

  step = phase_difference_table[ phase ][ dev->last_phase ];

  // If the difference is ambiguous, add the maximum change in the
  // current direction.  If the current direction is undefined, ignore
  // the transition to avoid biasing the drift in any particular
  // direction.
  if ( step == 2 ) {
    if ( dev->last_step < 0 )      step = -2;
    else if ( dev->last_step > 0 ) step = 2;
    else step = 0;
  }

  dev->position  += step;   // accumulate the position count
  dev->last_step  = step;   // record the state machine state
  dev->last_phase = phase;
}

/****************************************************************/
// This is called from interrupt context. It polls each encoder in turn.
void encoder_counter_poll(void)
{
  poll_encoder( &encoder1 );
  poll_encoder( &encoder2 );

#if CONFIG_ENCODER_ON_EXPANSION
  poll_encoder( &extra_encoder );
#endif

}

/****************************************************************/
