% Power iteration clustering using similarity kernels, can use precalcuated
% normalizers multiple experiments on the same dataset
%
% Input:
% M - row-instance feature matrix
% v0 - starting vector
% conv - convergence threshold
% maxit - maximum number of iterations
% kernel - name of the similarity kernel:
%  'inner' - inner product similarity
%  'bipart' - bipartite graph walk similarity
%  'cos' - cosine similarity
% normalizers - optional cell array of cached normalizers for the kernel
%
% Output:
% vt - 1-d PIC embedding
% i - iterations ran
% t - runtime
%
% Author: Frank Lin (frank@cs.cmu.edu)

function [vt,i,t]=pic_kernel(M,v0,conv,maxit,kernel,normalizers)

switch kernel
    case 'inner'
        fprintf('using inner product kernel...');
        if length(normalizers)==1
            fprintf('pre-calculated normalizer(s) found\n');
            N=normalizers{1};
        else
            fprintf('calculating normalizer(s)...');
            N=prep_inner(M);
            fprintf('done\n');
        end
    case 'bipart'
        fprintf('using bipartite graph walk kernel...');
        if length(normalizers)==2
            fprintf('pre-calculated normalizer(s) found\n');
            Nrow=normalizers{1};
            Ncol=normalizers{2};
        else
            fprintf('calculating normalizer(s)...');
            [Nrow,Ncol]=prep_bipart(M);
            fprintf('done\n');
        end
    case 'cos'
        fprintf('using cosine similarity kernel...');
        if length(normalizers)==2
            fprintf('pre-calculated normalizer(s) found\n');
            N=normalizers{1};
            D=normalizers{2};
        else
            fprintf('calculating normalizer(s)...');
            [N,D]=prep_cos(M);
            fprintf('done\n');
        end 
    otherwise
        fprintf('not using kernels\n')
end

n=size(M,1);

vt=v0;
dt=ones(n,1);
dtp=zeros(n,1);

i=0;
tic;
while(max(abs(dt-dtp))>conv&&i<maxit)
    
    vtp=vt;
    dtp=dt;
    
    switch kernel
        case 'inner'
            vt=N*(M*(M'*vt));
        case 'bipart'
            vt=Nrow*(M*(Ncol*(M'*vt)));
        case 'cos'
            vt=N*(D*(M*(M'*(D*vt))));
        otherwise
            vt=M*vt;
    end

    vt=vt/sum(vt);
    dt=abs(vt-vtp);
    
    i=i+1;
    
end
t=toc;

end