% wvRN relational learning algorithm from:
%
% Classification in Networked Data: A toolkit and a univariate case study,
% Sofus A. Macskassy and Foster Provost. JMLR 2007.
%
% Solution is equivalent to a basic multi-class version of:
%
% Semi-Supervised Learning Using Gaussian Fields and Harmonic Functions,
% Xiaojin Zhu, John Lafferty, and Zoubin Ghahramani. ICML 2003.
%
% Input:
% P - a column-normalized transition matrix
% L - row-instance seed labels (1 or 0)
% conv - convergence threshold
% maxit - maximum number of iterations
% cmn - whether to perform class mass normalization (Zhu et al. 2003)
%
% Output:
% X - row-instance predicted soft labels
% i - iterations ran
%
% Author: Frank Lin (frank@cs.cmu.edu)

function [X,i]=harmonic(P,L,conv,maxit,cmn)

% keep track of seed indices
s=find(sum(L,2));

% for iteration
Xp=zeros(size(L));
X=full(L);

i=0;
while(max(max(abs(X-Xp)))>conv&&i<maxit) 
    % save previous
    Xp=X;
    % propagate
    X=(X'*P)';
    % normalize per-instance
    X=normrow(X);
    % reinstate seed labels
    X(s,:)=L(s,:);
    % increment counter
    i=i+1;
end

if cmn
    % get estimated class priors
    p=sum(L);
    p=p./sum(p);
    % calculate unlabeled class mass
    m=sum(X-L);
    % apply to results
    X=X*diag(p./m);
end

end