(* Dynamics *)
(* implements reduction on configuration where all possible
 * steps are taken in each round
 *)

signature DYNAMICS =
sig
    exception DynamicError of string (* should be impossible for well-typed programs *)

    datatype config = Config of IntSyn.proc list * int

    val split : IntSyn.chan -> IntSyn.proc list -> IntSyn.proc list
                -> (IntSyn.proc list * IntSyn.proc * IntSyn.proc list) option
    val iterate : IntSyn.env -> config -> config

    val exec : IntSyn.env -> IntSyn.procname -> unit (* print result if verbosity >= 1 *)
end

structure Dynamics :> DYNAMICS =
struct

structure I = IntSyn

datatype config = Config of I.proc list * int
(* Config([P1,...,Pn], n) where n is the next unused channel number *)
(* the configuration C = P1,...,Pn is in reverse order from the
 * the typing derivation in that the client Pi precedes the provider Pj
 * in the list order (that is, i < j)
 * 
 * We assume and maintain that . |- Pn, P(n-1), ..., P1 :: Delta
 * that is the configuration is closed and well-typed
 *)
(* should not arise for well-typed configurations *)
exception DynamicError of string

fun opt2list (NONE) = []
  | opt2list (SOME(P)) = [P]

fun select k ((l,P)::branches) = if k = l then P else select k branches
  | select k nil = raise DynamicError "label not recognized"

fun pp_proc (I.Send(x, V)) = "send " ^ I.pp_chan x ^ " " ^ I.pp_value V
  | pp_proc (I.Recv(x, K)) = "recv " ^ I.pp_chan x ^ " ..."
  | pp_proc (I.Fwd(x, y)) = "fwd " ^ I.pp_chan x ^ " " ^ I.pp_chan y
  | pp_proc (I.Call(p, x, ys)) = "call " ^ p ^ " " ^ I.pp_chan x ^ " " ^ String.concatWith " " (List.map I.pp_chan ys)
  | pp_proc (I.Cut(x, A_opt, P, Q)) = I.pp_chan x ^ " <- ... ; ..."
  | pp_proc (I.Marked(marked_P)) = pp_proc (Mark.data marked_P)

fun pp_config C = String.concatWith "\n" (List.map pp_proc C) ^ "\n% -----\n"

fun get_channel_opt (I.Channel(x)) = SOME(x)
  | get_channel_opt (I.MarkedValue(marked_V)) = get_channel_opt (Mark.data marked_V)
  | get_channel_opt V = NONE

fun get_channel V =
    (case get_channel_opt V
      of SOME(y) => y
       | NONE => raise Match)

fun project_label ((I.Label(k,Vk), P)::K) l =
    if k = l
    then (Vk,P)::project_label K l
    else project_label K l
  | project_label ((I.MarkedValue(marked_V), P)::K) l =
    project_label ((Mark.data marked_V, P)::K) l
  | project_label ((V,P)::K) l = project_label K l
  | project_label nil l = nil

fun project_pair ((I.Pair(V1,V2), P)::K) NONE nil =
    let val y = get_channel V1
    in project_pair K (SOME(y)) [(V2,P)] end
  | project_pair ((I.Pair(V1,V2), P)::K) (SOME(y)) Kpairs =
    let val z = get_channel V1
        val P' = I.subst_proc [(y,z)] P
    in project_pair K (SOME(y)) (Kpairs @ [(V2,P')]) end
  | project_pair ((I.MarkedValue(marked_V), P)::K) yOpt Kpairs =
    project_pair ((Mark.data marked_V, P)::K) yOpt Kpairs
  | project_pair nil (SOME(y)) Kpairs = (y, Kpairs)

fun project_unit ((I.Unit, P)::K) = P
  | project_unit ((I.MarkedValue(marked_V), P)::K) =
    project_unit ((Mark.data marked_V, P)::K)

(* assumes no conflict *)
fun subst_cont theta ((V,P)::K) = (* V binds variables *)
    (V, I.subst_proc theta P)::subst_cont theta K
  | subst_cont theta nil = nil

(* need two reduce functions: one for positive, one for negative types *)
(* the first case only works if the channel a is linear! *)
fun reducePos a (I.MarkedValue(marked_V)) K = reducePos a (Mark.data marked_V) K
  | reducePos a V [(I.MarkedValue(marked_V'),P)] = reducePos a V [(Mark.data marked_V', P)]
  | reducePos a (I.Channel(a')) [(I.Channel(x'), P)] = [I.subst_proc [(a',x')] P] (* equal *)
  | reducePos a (I.Channel(a')) K = [I.Recv(a',I.Cont(K))]                        (* pattern V' is deeper *)
  | reducePos a V [(I.Channel(x'), P)] = [I.Send(a,V), I.subst_proc [(a,x')] P]   (* msg V is deeper *)
  | reducePos a V K = reducePos' a V K
(* V not a mark or channel *)
and reducePos' a (I.Label(k,V)) K = reducePos a V (project_label K k)
  | reducePos' a (I.Pair(V1,V2)) K =
    let val (y, K') = project_pair K NONE nil
        val b = get_channel V1
    in reducePos a V2 (subst_cont [(b,y)] K') end
  | reducePos' a (I.Unit) K = [project_unit K]

fun reduceNeg a K (I.MarkedValue(marked_V)) = reduceNeg a K (Mark.data marked_V)
  | reduceNeg a [(I.MarkedValue(marked_V'), P)] V = reduceNeg a [(Mark.data marked_V', P)] V
  | reduceNeg a [(I.Channel(x'),P)] (I.Channel(a')) = [I.subst_proc [(a',x')] P] (* equal *)
  | reduceNeg a K                   (I.Channel(a')) = [I.Recv(a',I.Cont(K))]     (* pattern V' is deeper *)
  | reduceNeg a [(I.Channel(x'),P)] V = [I.subst_proc [(a,x')] P, I.Send(a,V)]   (* msg V is deeper *)
  | reduceNeg a K V = reduceNeg' a K V
(* V not a mark or channel *)
and reduceNeg' a K (I.Label(k,V)) = reduceNeg a (project_label K k) V
  | reduceNeg' a K (I.Pair(V1,V2)) =
    let val (y, K') = project_pair K NONE nil
        val b = get_channel V1
    in reduceNeg a (subst_cont [(b,y)] K') V2 end
  | reduceNeg' a K (I.Unit) = [project_unit K]

(* split a (C0 @ [P] @ C2) C1rev = SOME(rev C0 @ C1rev, P, C2)
 * where P communicates along a
 * = NONE, if now such process P exists
 *)
fun split a (I.Send(b,V)::C2) C1rev =
    if a = b then SOME(C1rev, I.Send(b,V), C2)
    else split a C2 (I.Send(b,V)::C1rev)
  | split a (I.Recv(b,I.Cont(K))::C2) C1rev =
    if a = b then SOME(C1rev, I.Recv(b,I.Cont(K)), C2)
    else split a C2 (I.Recv(b,I.Cont(K))::C1rev)
  | split a (I.Marked(marked_P)::C2) C1rev =
    split a (Mark.data marked_P::C2) C1rev
  | split a (P::C2) C1rev = split a C2 (P::C1rev) (* ignore P = fwd, call, cut *)
  | split a nil C1rev = NONE

(* step_par env n changed C C' = (n', changed', rev D @ C') if C --> D in a parallel step
 *
 * The input C is already "reversed" with the "rightmost" process in a configuration
 * according to configuration typing comes first, so it is easy to lookup up
 * processes in the remainder of the input configuration.
 *
 * n is the next available free channel index in C
 * n' is the next available free channel index in rev D @ C'
 * changed indicates if a change reduction took place already in the parallel reduction
 * changed' is true if either changed is true or D is different from C
 *)

fun step_par env n changed (I.Recv(a,I.Cont(K))::C) C' =  (* positive: receive from the left *)
    (case split a C nil
      of SOME(C1rev, I.Send(a,V), C2) => (* P :: a; move "right" *)
         step_par env n true (List.revAppend(C1rev, C2)) (reducePos a V K @ C') (* V first, then K *)
       | NONE => step_par env n changed C (I.Recv(a,I.Cont(K))::C'))

  | step_par env n changed (I.Send(a,V)::C) C' = (* negative: send to left *)
    (case split a C nil
      of SOME(C1rev, I.Recv(a,I.Cont(K)), C2) => (* P :: a; move "right" *)
         step_par env n true (List.revAppend(C1rev, C2)) (reduceNeg a K V @ C') (* K first, then V *)
       | NONE => step_par env n changed C (I.Send(a,V)::C'))

  | step_par env n changed (I.Fwd(a,b)::C) C' = (* or: substitute everywhere in config *)
    (case split b C nil
      of SOME(C1rev, P, C2) => (* P :: b, positive or negative *)
         step_par env n true (List.revAppend(C1rev, C2)) (I.subst_proc [(a,b)] P::C')
       | NONE => step_par env n changed C (I.Fwd(a,b)::C'))

  | step_par env n changed (I.Cut(x, _, P, Q)::C) C' =
    let val a = I.Chan(n)       (* fresh channel *)
    in step_par env (n+1) true C (I.subst_proc [(a,x)] P::I.subst_proc [(a,x)] Q::C') end

  | step_par env n changed (I.Call(p, a, bs)::C) C' =
    let val SOME(I.ProcDef(p, I.Tp(x,nA), yBs, P, ext)) = I.lookup_procname env p (* must be defined *)
        val ys = List.map (fn I.Tp(y,nB) => y) yBs
        val P' = I.subst_proc (ListPair.zip (a::bs, x::ys)) P
    in step_par env n true C (P'::C') end

  | step_par env n changed (I.Marked(marked_P)::C) C' =
    step_par env n changed (Mark.data marked_P::C) C'

  | step_par env n changed nil C' = (changed, Config(List.rev C', n))

(* iterate env (Config(C,n)) = Config(C',n')
 * if C -->* C' and C' is final, where --> is a single parallel step
 * n is the next available channel index in C
 * n' is the next available channel index in C'
 * in parallel reduction, C = C' is possible, in which case we terminate
 *)
fun iterate env (Config(C,n)) =
    let (* val () = print ("before:\n" ^ pp_config C) *)
        val (changed, Config(C',n')) = step_par env n false C nil
        (* val () = print ("after:\n" ^ pp_config C') *)
    in
        if changed then iterate env (Config(C',n'))
        else Config(C',n') (* C = C' *)
    end

fun pp_channel (I.Channel(b)) = I.pp_chan b
  | pp_channel (I.MarkedValue(marked_V)) = pp_channel (Mark.data marked_V)

fun pp_prefix (I.Label(k,V)) = I.pp_tag k ^ "." ^ pp_prefix V
  | pp_prefix (I.Pair(V1,V2)) = pp_channel V1 ^ "." ^ pp_prefix V2
  | pp_prefix (I.Unit) = "()"
  | pp_prefix (I.Channel(a')) = ""
  | pp_prefix (I.MarkedValue(marked_V)) = pp_prefix (Mark.data marked_V)

fun cont_chan (I.Unit) = nil
  | cont_chan (I.Label(k,V)) = cont_chan V
  | cont_chan (I.Pair(V1,V2)) = cont_chan V2
  | cont_chan (I.Channel(a')) = [a']
  | cont_chan (I.MarkedValue(marked_V)) = cont_chan (Mark.data marked_V)

fun cont_chan1 V = case cont_chan V of [a'] => a' | _ => raise Match

(* depth calculation here accurate? *)
fun msg_chans (I.Unit) depth = nil 
  | msg_chans (I.Label(k,V)) depth = msg_chans V (depth-1)
  | msg_chans (I.Pair(V1,V2)) depth = (cont_chan1 V1, depth)::msg_chans V2 (depth-1) (* V1 = y *)
  | msg_chans (I.Channel(a')) depth = nil
  | msg_chans (I.MarkedValue(marked_V)) depth = msg_chans (Mark.data marked_V) depth

(* observe env a [b1,...,bn] (n,C) = str
 * observe_frontier env [b1,...,bn] (n,C) = str
 * 
 * interact with the configuration to receive messages
 * along channels a, b1, ..., bn until a negative type
 * is encountered or the channel is closed with the '()' message
 *
 * n is the next available channel index in C for fresh allocation
 *)
fun observe env (a,depth) frontier (Config(C,n)) = (* depth <> 0 *)
    ((* print (pp_config C) ; *)
     case split a C nil
      of SOME(C1rev, I.Recv(a, I.Cont(K)), C2) => (* negative *)
         "-\n" ^ observe_frontier env frontier (Config(List.revAppend(C1rev, C2),n))
       | SOME(C1rev, I.Send(a, V), C2) =>
         (case cont_chan V
           of nil => pp_prefix V ^ "\n" ^ observe_frontier env (msg_chans V (depth-1) @ frontier)
                                                           (Config(List.revAppend(C1rev, C2), n))
            | [a'] => pp_prefix V ^ next env (a',depth-1) (msg_chans V (depth-1) @ frontier)
                                         (Config(List.revAppend(C1rev, C2), n)))
       | NONE => raise DynamicError ("channel " ^ I.pp_chan a ^ " undefined"))

and next env (a,0) frontier (Config(C,n)) =
    "?\n" ^ observe_frontier env frontier (Config(C,n))
  | next env (a,depth) frontier (Config(C,n)) =
    (* no need to iterate again under asynchronous communication *)
    observe env (a,depth) frontier (Config(C,n)) (* (iterate env (Config(C,n))) *)
and observe_frontier env nil (Config(C,n)) = ""
  | observe_frontier env ((b,depth)::bs) (Config(C,n)) =
    I.pp_chan b ^ " -> " ^ next env (b,depth) bs (Config(C,n))

(* exec env p = ()
 * prints the messages sequence in channels visible
 * at the interface to process p, which must be closed except
 * for the channel provided
 *)
fun exec env p =
    let val SOME(I.ProcDef(p, I.Tp(x,nA), nil, P, ext)) = I.lookup_procname env p (* must be defined *)
        val P' = I.subst_proc [(I.Chan(0), x)] P
        val depth = case !Flags.depth of NONE => ~1 | SOME(n) => n (* negative = no bound *)
        val finalConfig = iterate env (Config([P'],1)) (* 0 already used *)
        val finalString = observe_frontier env [(I.Chan(0),depth)] finalConfig (* 0 already used *)
        val () = if !Flags.verbosity >= 1 then print (finalString) else ()
    in () end

end (* structure Dynamics *)
