
structure IntSyn =
struct

type tpname = string
type label = string
type procname = string
type varname = string
type ext = Mark.ext option      (* option extent = source region info *)

datatype chan = Var of varname
              | Chan of int

datatype tp = Plus of (label * tpname) list
            | Tensor of tpname * tpname
            | One
            | With of (label * tpname) list
            | Lolli of tpname * tpname

datatype msg = Label of label
             | Unit
             | Channel of chan

datatype proc = Send of chan * msg * proc option
              | Recv of chan * match
              | Fwd of chan * chan
              | Call of procname * chan * chan list
              | Cut of chan * tpname option * proc * proc
              | Marked of proc Mark.marked

     and match = MatchLabel of (label * proc) list
               | MatchUnit of proc
               | MatchChannel of chan * proc

datatype parm = Tp of chan * tpname

datatype status = Must | May | Used | Right

datatype typing = TpN of chan * tpname * status

datatype def = TpDef of tpname * tp * ext
             | ProcDef of procname * parm * parm list * proc * ext
             | Exec of procname * ext
             | Fail of def * ext
             | Error of ext

type env = def list

fun lookup_procname ((d as ProcDef(q, _, _, _, _))::env) p =
    if p = q then d else lookup_procname env p
  | lookup_procname (_::env) p = lookup_procname env p
  | lookup_procname nil p = raise Match (* fix *)

fun lookup_tpn x (TpN(y, nB, s)::Delta) =
    if x = y then SOME(TpN(y, nB, s)) else lookup_tpn x Delta
  | lookup_tpn x nil = NONE

fun expand_opt (TpDef(nB, B, ext)::env) nA =
    if nB = nA then SOME(B) else expand_opt env nA
  | expand_opt (_::env) nA = expand_opt env nA
  | expand_opt nil nA = NONE

fun expand env nA =
    (case expand_opt env nA
      of SOME(A) => A
       | NONE => raise Match)

fun select (k:label) ((l,elem)::alts) =
    if k = l then SOME(elem) else select k alts
  | select k nil = NONE

fun pp_tag k = "'" ^ k

fun pp_chan (Var(x)) = x
  | pp_chan (Chan(a)) = "(" ^ Int.toString a ^ ")"

fun pp_msg (Unit) = "()"
  | pp_msg (Label(k)) = k
  | pp_msg (Channel(a)) = pp_chan a

fun is_internal nA = (String.sub(nA, 0) = #"%")

fun parens true str = "(" ^ str ^ ")"
  | parens false str = str

fun pp_tpn env par nA =
    if is_internal nA
    then pp_tp env par (expand env nA)
    else nA
and pp_tp env par (Plus(alts)) = "+" ^ pp_alts env alts
  | pp_tp env par (One) = "1"
  | pp_tp env par (With(alts)) = "&" ^ pp_alts env alts
  | pp_tp env par (Tensor(nA,nB)) = parens par (pp_tpn env true nA ^ " * " ^ pp_tpn env true nB)
  | pp_tp env par (Lolli(nA,nB)) = parens par (pp_tpn env true nA ^ " -o " ^ pp_tpn env true nB)
and pp_alts env alts = "{"
                   ^ String.concatWith ", " (List.map (fn (k,nAk) => pp_tag k ^ " : " ^ pp_tpn env false nAk) alts)
                   ^ "}"

val pp_tpn = fn env => fn nA => pp_tpn env false nA
val pp_tp = fn env => fn A => pp_tp env false A

type subst = (chan * chan) list

(* substitutions are only (x/x) or (a/x) or (a/b)
 * so no capture can occur, only shadowing
 *)

fun subst_chan nil x = x
  | subst_chan ((b:chan,y)::theta) x =
    if x = y then b
    else subst_chan theta x

fun subst_msg theta Unit = Unit
  | subst_msg theta (Label(k)) = Label(k)
  | subst_msg theta (Channel(x)) = Channel(subst_chan theta x)

fun subst_proc theta (Send(x,Unit,NONE)) =
    Send(subst_chan theta x, Unit, NONE)
  | subst_proc theta (Send(x,msg,SOME(P))) =
    Send(subst_chan theta x, subst_msg theta msg, SOME(subst_proc theta P))
  | subst_proc theta (Recv(x,match)) =
    Recv(subst_chan theta x, subst_match theta match)
  | subst_proc theta (Fwd(x,y)) =
    Fwd(subst_chan theta x, subst_chan theta y)
  | subst_proc theta (Call(p, x, ys)) =
    Call(p, subst_chan theta x, List.map (subst_chan theta) ys)
  | subst_proc theta (Cut(x, A_opt, P, Q)) =
    Cut(x, A_opt, subst_proc ((x,x)::theta) P, subst_proc ((x,x)::theta) Q)
  | subst_proc theta (Marked(marked_P)) =
    subst_proc theta (Mark.data marked_P)
and subst_match theta (MatchLabel(alts)) = MatchLabel(List.map (fn (l,Pl) => (l, subst_proc theta Pl)) alts)
  | subst_match theta (MatchUnit(P)) = MatchUnit(subst_proc theta P)
  | subst_match theta (MatchChannel(x,P)) = MatchChannel(x, subst_proc ((x,x)::theta) P)

fun eq_shallow (Tensor(nA1,nA2)) (Tensor(nB1,nB2)) = nA1 = nB1 andalso nA2 = nB2
  | eq_shallow (One) (One) = true
  | eq_shallow (Lolli(nA1,nA2)) (Lolli(nB1,nB2)) = nA1 = nB1 andalso nA2 = nB2
  | eq_shallow (Plus(alts)) (Plus(blts)) = eq_alts alts blts
  | eq_shallow (With(alts)) (With(blts)) = eq_alts alts blts
  | eq_shallow A B = false
and eq_alts alts blts =
    List.all (fn (k,nAk) => List.exists (fn (l,nBl) => k = l andalso nAk = nBl) blts) alts
    andalso List.all (fn (l,nBl) => List.exists (fn (k,nAk) => l = k andalso nAk = nBl) alts) blts
                                           
fun abbrev_tp env A =
    (case List.find (fn TpDef(nB, B, _) => eq_shallow A B | _ => false) env
      of SOME(TpDef(nB, _, _)) => SOME(nB)
       | NONE => NONE)

fun sub env memo (Tensor(nA1,nA2)) (Tensor(nB1,nB2)) =
    subn env memo nA1 nB1 andalso subn env memo nA2 nB2
  | sub env memo (One) (One) = true
  | sub env memo (Lolli(nA1,nA2)) (Lolli(nB1,nB2)) =
    subn env memo nB1 nA1 andalso subn env memo nA2 nB2
  | sub env memo (Plus(alts)) (Plus(blts)) =
    List.all (fn (k,nAk) => List.exists (fn (l,nBl) => k = l andalso subn env memo nAk nBl) blts) alts
  | sub env memo (With(alts)) (With(blts)) =
    List.all (fn (l,nBl) => List.exists (fn (k,nAk) => l = k andalso subn env memo nAk nBl) alts) blts
  | sub env memo A B = false

and subn env memo nA nB =
    List.exists (fn (nA',nB') => nA = nA' andalso nB = nB') memo
    orelse sub env ((nA,nB)::memo) (expand env nA) (expand env nB)

fun eqtp env nA nB = subn env nil nA nB andalso subn env nil nB nA

end (* structure IntSyn *)
