structure Dynamics =
struct

structure I = IntSyn

exception DynamicError of string

type config = I.proc list

fun opt2list (NONE) = []
  | opt2list (SOME(P)) = [P]

fun select k ((l,P)::branches) = if k = l then P else select k branches
  | select k nil = raise DynamicError "label not recognized"

fun pp_proc (I.Send(x, msg, _)) = "send " ^ I.pp_chan x ^ " " ^ I.pp_msg msg ^ " ..."
  | pp_proc (I.Recv(x, match)) = "recv " ^ I.pp_chan x ^ " ..."
  | pp_proc (I.Fwd(x, y)) = "fwd " ^ I.pp_chan x ^ " " ^ I.pp_chan y
  | pp_proc (I.Call(p, x, ys)) = "call " ^ p ^ " " ^ I.pp_chan x ^ " " ^ String.concatWith " " (List.map I.pp_chan ys)
  | pp_proc (I.Cut(x, A_opt, P, Q)) = I.pp_chan x ^ " <- ... ; ..."
  | pp_proc (I.Marked(marked_P)) = pp_proc (Mark.data marked_P)

fun pp_config C = String.concatWith "\n" (List.map pp_proc C) ^ "\n% -----\n"

fun reduce (I.Label(k)) (I.MatchLabel(branches)) = select k branches
  | reduce (I.Unit) (I.MatchUnit(P)) = P
  | reduce (I.Channel(a)) (I.MatchChannel(x,P)) = I.subst_proc [(a,x)] P
  | reduce msg match = raise DynamicError "message / match mismatch"

fun split a (I.Send(b,msg,Popt)::C2) C1rev =
    if a = b then SOME(C1rev, I.Send(b,msg,Popt), C2)
    else split a C2 (I.Send(b,msg,Popt)::C1rev)
  | split a (I.Recv(b,match)::C2) C1rev =
    if a = b then SOME(C1rev, I.Recv(b,match), C2)
    else split a C2 (I.Recv(b,match)::C1rev)
  | split a (I.Marked(marked_P)::C2) C1rev =
    split a (Mark.data marked_P::C2) C1rev
  | split a (P::C2) C1rev = split a C2 (P::C1rev) (* ignore P = fwd, call, cut *)
  | split a nil C1rev = NONE

(* keep the state in reverse order *)

fun step_par env n changed (I.Recv(a,match)::C) C' =  (* positive: receive from the left *)
    (case split a C nil
      of SOME(C1rev, I.Send(a,msg,Popt), C2) => (* P :: a; move "right" *)
         step_par env n true (List.revAppend(C1rev, C2)) ([reduce msg match] @ opt2list Popt @ C')
       | NONE => step_par env n changed C (I.Recv(a,match)::C'))

  | step_par env n changed (I.Send(a,msg,Popt)::C) C' = (* negative: send to left *)
    (case split a C nil
      of SOME(C1rev, I.Recv(a,match), C2) => (* P :: a; move "right" *)
         step_par env n true (List.revAppend(C1rev, C2)) ([reduce msg match] @ opt2list Popt @ C')
       | NONE => step_par env n changed C (I.Send(a,msg,Popt)::C'))

  | step_par env n changed (I.Fwd(a,b)::C) C' = (* or: substitute everywhere in config *)
    (case split b C nil
      of SOME(C1rev, P, C2) => (* P :: b, positive or negative *)
         step_par env n true (List.revAppend(C1rev, C2)) (I.subst_proc [(a,b)] P::C')
       | NONE => step_par env n changed C (I.Fwd(a,b)::C'))

  | step_par env n changed (I.Cut(x, _, P, Q)::C) C' =
    let val a = I.Chan(n)       (* fresh channel *)
    in step_par env (n+1) true C (I.subst_proc [(a,x)] P::I.subst_proc [(a,x)] Q::C') end

  | step_par env n changed (I.Call(p, a, bs)::C) C' =
    let val I.ProcDef(p, I.Tp(x,nA), yBs, P, ext) = I.lookup_procname env p
        val ys = List.map (fn I.Tp(y,nB) => y) yBs
        val P' = I.subst_proc (ListPair.zip (a::bs, x::ys)) P
    in step_par env n true C (P'::C') end

  | step_par env n changed (I.Marked(marked_P)::C) C' =
    step_par env n changed (Mark.data marked_P::C) C'

  | step_par env n changed nil C' = (n, changed, List.rev C')

fun iterate env (n, C) =
    let (* val () = print ("before: " ^ pp_config C) *)
        val (n', changed, C') = step_par env n false C nil
        (* val () = print ("after: " ^ pp_config C') *)
    in
        if changed then iterate env (n', C')
        else (n', C') (* C = C' *)
    end

fun pp_msg (I.Chan(i)) msg = Int.toString i ^ " = " ^ I.pp_msg msg

fun observe env a frontier (n,C) =
    ((* print (pp_config C) ; *)
     case split a C nil
      of SOME(C1rev, I.Recv(a,match), C2) => (* negative *)
         " ---- \n" ^ observe_frontier env frontier (n, List.revAppend(C1rev, C2))
       | SOME(C1rev, I.Send(a, msg as I.Unit, NONE), C2) =>
         I.pp_msg msg ^ "\n" ^ observe_frontier env frontier (n, List.revAppend(C1rev, C2))
       | SOME(C1rev, I.Send(a, msg as I.Label _, SOME(P)), C2) =>
         I.pp_msg msg ^ "." ^ next env a frontier (n, P::List.revAppend(C1rev, C2))
       | SOME(C1rev, I.Send(a, msg as I.Channel(b), SOME(P)), C2) =>
         I.pp_msg msg ^ "." ^ next env a (b::frontier) (n, P::List.revAppend(C1rev, C2))
       | NONE => raise DynamicError ("channel " ^ I.pp_chan a ^ " undefined"))

and next env a frontier (n,C) =
    observe env a frontier (iterate env (n,C))
and observe_frontier env nil (n,C) = ""
  | observe_frontier env (b::bs) (n,C) =
    I.pp_chan b ^ " -> " ^ observe env b bs (iterate env (n,C))

fun exec env p =
    let val I.ProcDef(p, I.Tp(x,nA), nil, P, ext) = I.lookup_procname env p
        val P' = I.subst_proc [(I.Chan(0), x)] P
        val final = observe_frontier env [I.Chan(0)] (1,[P']) (* 0 already used *)
        val () = if !Flags.verbosity >= 1 then print (final) else ()
    in () end

end (* structure Dynamics *)
