%{
(** ND Parser *)

    let errors = Error_msg.create ()
%}

%token <int> NAT
%token <string> IDENT
%token <string> LABEL

%token MATCH WITH END
%token TYPE DEFN FAIL
%token PROC
%token READ WRITE CUT ID CALL REUSE
%token VALUE

%token COMMA COLON
%token LPAREN RPAREN
%token LBRACE RBRACE
%token RIGHTARROW EQUAL BAR
%token STAR PLUS
%token EOF

(* precedence & associativity *)
%right STAR

%right COMMA
%right LABEL

%start prog

(* types *)
%type <Ast.env> prog
%type <Ast.defn> defn
%type <Ast.exp> exp
%type <(Ast.pat * Ast.exp) list> branches
%type <Ast.pat * Ast.exp> branch
%type <Ast.exp> atom
%type <Ast.exp list> atoms
%type <Ast.pat> pat
%type <Ast.varname * Ast.tp> parm
%type <(Ast.varname * Ast.tp) list> parms
%type <Ast.tp> tp
%type <(Ast.label * Ast.tp) list> alts
%type <Ast.label * Ast.tp> alt

%%

prog :
  | d = defn; defns = prog; { d::defns }
  | EOF;                    { [] }

defn :
  | TYPE; a = IDENT; EQUAL; tau = tp;
    { Ast.TypeDefn(a, tau) }

  | DEFN; f = IDENT; parms = parms; COLON; tau = tp; EQUAL; e = exp;
    { Ast.ExpDefn(f, parms, tau, e) }

exp :
  | e = atom;                  { e }
  | f = IDENT; args = atoms;   { Ast.Call(f, args) }
  | e1 = exp; COMMA; e2 = exp; { Ast.Pair(e1, e2) }
  | k = LABEL; e = exp;        { Ast.Inj(k, e) }
  | MATCH; e = exp; WITH;
    branches = branches;
    END;                      { Ast.MatchWith(e, branches) }

atom :
  | x = IDENT;                { Ast.Var(x) }
  | LPAREN; RPAREN;           { Ast.Unit } 
  | LPAREN; e = exp ; RPAREN; { e }

atoms :
  | e = atom;            { [e] }
  | e = atom; es = atoms { e::es }

branches :
  | branch = branch;
    branches = branches;
    { branch::branches }
  | (* empty *)
    { [] }

branch :
  | BAR; p = pat; RIGHTARROW; e = exp; { (p, e) }

pat :
  | p1 = pat; COMMA; p2 = pat;  { Ast.PairPat(p1, p2) }
  | LPAREN; RPAREN;             { Ast.UnitPat }
  | k = LABEL; p = pat;         { Ast.InjPat(k, p) }
  | x = IDENT;                  { Ast.VarPat(x) }
  | LPAREN; p = pat; RPAREN;    { p }

parm :
  | LPAREN; x = IDENT; COLON; tau = tp; RPAREN; { (x, tau) }

parms :
  | parm = parm; parms = parms; { parm::parms }
  | (* empty *)                 { [] }

tp :
  | PLUS; LBRACE; alts = alts; RBRACE; { Ast.Plus(alts) }
  | tau1 = tp; STAR; tau2 = tp; { Ast.Times(tau1, tau2) }
  | n = NAT;
    { if n = 1 then Ast.One
      else ( Error_msg.error errors None "only '1' is a type" ; raise Error_msg.Error ) }
  | a = IDENT; { Ast.TpName(a) }
  | LPAREN; tp = tp; RPAREN; { tp }

alts :
  | alt = alt; { [alt] }
  | alt = alt; COMMA; alts = alts; { alt::alts }

alt :
  | l = LABEL; COLON; tau = tp; { (l, tau) }

%%
