{
(** Sax Lexer *)

module N = Nd_parser

let errors = Error_msg.create ()

let text = Lexing.lexeme

let from_lexbuf : Lexing.lexbuf -> Mark.src_span option =
  fun lexbuf ->
    Mark.of_positions
      (Lexing.lexeme_start_p lexbuf)
      (Lexing.lexeme_end_p lexbuf)
    |> Option.some

let error lexbuf (msg : string) =
  let src_span = from_lexbuf lexbuf in
  Error_msg.error errors src_span msg

}

let idstart = ['a'-'z' 'A'-'Z' '_']
let idchar = ['a'-'z' 'A'-'Z' '_' '0'-'9'] (* omitting '$' from Sax *)
let ident = idstart idchar*
let label = '\'' idchar+

let digit = ['0'-'9']
let num = digit*


let ws = [' ' '\t' '\r']

rule initial = parse
  | ws+  { initial lexbuf }
  | '\n' { Lexing.new_line lexbuf;
           initial lexbuf
         }
  | "%" { comment_line lexbuf }
  | "(*" { comment_block 1 lexbuf }

  | ',' { N.COMMA }
  | ':' { N.COLON }

  | '(' { N.LPAREN }
  | ')' { N.RPAREN }
  | '{' { N.LBRACE }
  | '}' { N.RBRACE }

  | "=>" { N.RIGHTARROW }
  | '=' { N.EQUAL }
  | '|' { N.BAR }

  | '*' { N.STAR }
  | '+' { N.PLUS }

  | num as n { N.NAT (int_of_string n) }  

  | "type"   { N.TYPE }
  | "defn"   { N.DEFN }
  | "fail"   { N.FAIL } (* not parsed! *)
  | "match"  { N.MATCH }
  | "with"   { N.WITH }
  | "end"    { N.END }

  (* from *.sax, to avoid conflicts *)
  | "proc"   { N.PROC }
  | "read"   { N.READ }
  | "write"  { N.WRITE }
  | "cut"    { N.CUT }
  | "id"     { N.ID }
  | "call"   { N.CALL }
  | "reuse"  { N.REUSE }

  (* from *.sax.val *)
  | "value"  { N.VALUE }

  | label as name { N.LABEL name }
  | ident as name { N.IDENT name }

  | eof { N.EOF }

  | _  { error lexbuf
           (Printf.sprintf "Illegal character '%s'" (text lexbuf));
         initial lexbuf
       }

and comment_line = parse
  | '\n' { Lexing.new_line lexbuf; initial lexbuf }
  | eof  { N.EOF }
  | _ { comment_line lexbuf }

and comment_block depth = parse
  | '\n' { Lexing.new_line lexbuf; comment_block depth lexbuf }
  | "*)" { if depth = 1 then initial lexbuf else comment_block (depth-1) lexbuf }
  | "(*" { comment_block (depth+1) lexbuf }
  | _ { comment_block depth lexbuf }

{}
