(* Top Level *)

(* Use in the SML/NJ read/eval/print loop or build
 * a standalone executable with mlton
 *)

signature TOP =
sig
    val load_files : string list -> Ast.env (* only parses *)
    val sax : string -> OS.Process.status      (* Top.sax "<command line arguments>" = status *)
    val main : string * string list -> OS.Process.status (* for stand-alone executable *)
end (* signature TOP *)

structure Top :> TOP =
struct

structure G = GetOpt  (* from $/smlnj-lib/Util/getopt-sig.sml *)
structure A = Ast

(************************)
(* command line options *)
(************************)

datatype option =
         PrintDepth of string
       | Verbose of int
       | Help of bool

(* printing error/success messages to stdErr *)
fun say s = TextIO.output (TextIO.stdErr, s ^ "\n")

val usage =
    if "sml" = #file (OS.Path.splitDirFile (CommandLine.name ()))
    then "Top.sax \"<option>* <file>*\";"
    else "sax <option>* <file>*"
val header = Flags.version ^ "\n" ^ "Usage: " ^ usage ^ "\nwhere <option> is"
val options : option G.opt_descr list =
    [{short = "v", long = ["verbose"],
      desc = G.NoArg (fn () => Verbose(2)),
      help = "Give verbose status messages"},
     {short = "q", long = ["quiet"],
      desc = G.NoArg (fn () => Verbose(0)),
      help = "Run quietly"},
     {short = "d", long = ["debug"],
      desc = G.NoArg (fn () => Verbose(3)),
      help = "Print some debugging information"},
     {short = "h", long = ["help"],
      desc = G.NoArg (fn () => Help(true)),
      help = "Give short usage message and exit"},
     {short = "p", long = ["print_depth"],
      desc = G.ReqArg ((fn s => PrintDepth(s)), "<depth>"),
      help = "Maximal depth for printing values (-1 = no limit, default = 20)"}
    ]

val usage_info = G.usageInfo {header = header, options = options}

exception OS_FAILURE
exception OS_SUCCESS

fun exit_failure msg = ( say msg ; raise OS_FAILURE )
fun exit_success msg = ( say msg ; raise OS_SUCCESS )

fun get_options (args) =
    G.getOpt {argOrder = G.RequireOrder,
              options = options,
              errFn = exit_failure}
             args

fun process_option (PrintDepth(depth)) =
    (case Flags.parseInt(depth)
      of NONE => exit_failure ("maximal print depth '" ^ depth ^ "' not a number")
       | SOME(n) => Flags.printDepth := n)
  | process_option (Verbose(level)) = ( Flags.verbosity := level )
  | process_option (Help(true)) = exit_success usage_info
  | process_option (Help(false)) = ()

(*********************************)
(* loading and elaborating files *)
(*********************************)

fun readable file = OS.FileSys.access (file, [OS.FileSys.A_READ])

fun canonicalize file ext = OS.FileSys.realPath file
    handle OS.Path.Path => ErrorMsg.ERROR ext ("file " ^ file ^ " is not a valid path")

(* command line options applied before execution *)
fun apply_options line =
    let val args = String.tokens Char.isSpace line
        val (options, filenames) = get_options args (* may exit_failure(msgs) *)
        val () = List.app process_option options
        val () = case filenames
                  of nil => ()
                   | (_::_) => exit_failure ("spurious options: "
                                             ^ List.foldr (fn (arg,msg) => arg ^ " " ^ msg) "" filenames)
    in () end

(* load raw filenames = raw'
 * where raw' extends raw with declarations from filenames
 * may raise ErrorMsg.Error upon syntax error
 *)
fun load raw (file::filenames) =
    let
        val () = ParseState.reset ()
        val () = ErrorMsg.reset ()
        val () = if !Flags.verbosity >= 1
                 then TextIO.print ("% loading file " ^ file ^ "\n")
                 else ()
        val raw' = Parse.parse_sax file (* may raise ErrorMsg.Error *)
    in
        load (raw @ raw') filenames
    end
  | load raw nil = raw

fun load_files filenames = load nil filenames

fun exit_on_empty_files nil = exit_success Flags.version
  | exit_on_empty_files (_::_) = ()

fun last nil = "<bogusfile>"
  | last (file::nil) = file
  | last (file::filenames) = last filenames

(* main function to run file *)
fun test raise_exn args =
    (* reset flags *)
    let val () = Flags.reset ()
        (* get and apply options *)
        val (options, filenames) = get_options args
        (* val {base = _, ext = extOpt} = OS.Path.splitBaseExt (last filenames) *)
        val () = List.app process_option options
        val () = exit_on_empty_files filenames
        (* load files, returning an environment of declarations *)
        val raw = load_files filenames
            handle ErrorMsg.Error => exit_failure "% parsing failed"
                 | e => if raise_exn then raise e   (* for debugging purposes *)
                        else exit_failure "% internal error (uncaught exception)"
        val () = TextIO.print (Ast.pp_env raw ^ "\n")
    in
        exit_success "% success"
    end handle OS_SUCCESS => OS.Process.success
             | OS_FAILURE => OS.Process.failure

fun sax argstring = test true (String.tokens Char.isSpace argstring)
fun main (name, args) = test false args

end (* structure Top *)
