(* Parser *)
(* Author: Frank Pfenning <fp@cs.cmu.edu> *)

(*
 * Handwritten shift/reduce parser to support hand-tuned error messages.
 * Comments in the parser refer to the grammar in the readme file.
 *)

signature PARSE =
sig

    val parse_sax : string -> Ast.env (* may raise ErrorMsg.Error *)
    val parse_val : string -> Ast.valenv (* may raise ErrorMsg.Error *)

end  (* signature PARSE *)

structure Parse :> PARSE =
struct

structure A = Ast
structure PS = ParseState
structure M = Stream
structure T = Terminal
structure L = Lex

(******************)
(* Error Messages *)
(******************)
              
fun pp_tok t = "'" ^ T.toString t ^ "'"

fun pp_toks (nil) = ""
  | pp_toks (t::nil) = " or " ^ pp_tok t
  | pp_toks (t::ts) = pp_tok t ^ ", " ^ pp_toks ts

fun ^^(s1,s2) = s1 ^ "\n[Hint: " ^ s2 ^ "]"
infix ^^

exception ParseError of Lex.lexresult M.front

fun parse_error (region, msg) tf =
    ( ErrorMsg.error (PS.ext region) msg
    ; raise ParseError tf )

fun msg_expected t' t =
    ("expected " ^ pp_tok t' ^ ", found: " ^ pp_tok t)

fun error_expected (region, t', t) tf =
    ( ErrorMsg.error (PS.ext region) (msg_expected t' t)
    ; raise ParseError tf )

fun error_expected_h (region, t', t, error_hint) tf =
    ( ErrorMsg.error (PS.ext region) (msg_expected t' t ^^ error_hint)
    ; raise ParseError tf )

fun msg_expected_list ts t =
    "expected one of " ^ pp_toks ts ^ ", found: " ^ pp_tok t

fun error_expected_list (region, ts, t) tf =
    ( ErrorMsg.error (PS.ext region) (msg_expected_list ts t)
    ; raise ParseError tf )

fun error_expected_list_h (region, ts, t, error_hint) tf =
    ( ErrorMsg.error (PS.ext region) (msg_expected_list ts t ^^ error_hint)
    ; raise ParseError tf )
 
fun location (NONE) = "_"
  | location (SOME(mark)) = Mark.show(mark)

(*******************)
(* Data structures *)
(*******************)

type region = int * int
type prec = int                 (* precedence *)

(* stack items for shift/reduce parsing *)
datatype stack_item =
   Tok of T.terminal * region                          (* lexer token *)
 | Ids of string list * region                         (* identifiers *)
 | TpInfix of prec * (A.tp * A.tp -> A.tp) * region    (* infix type operator, all right assoc *)
 | TpAlts of (A.label * A.tp) list * region            (* tagged alternatives *)
 | Tp of A.tp * region                                 (* type *)
 | Parms of A.parm list * region                       (* typed parameters *)
 | Cmd of A.cmd * region                               (* expression *)
 | Pat of A.pat * region                               (* pattern *)
 | Branches of (A.pat * A.cmd) list * region           (* list of branches *)
 | Defn of A.defn * region                             (* top-level definition *)
 | Failure of region                                   (* negating a definition *)
 | Value of A.value * region                           (* large value, for result of evaluation *)
 | ValDefn of A.valdefn * region                       (* large value definition *)
 | Error                                               (* lexing or parsing error *)

datatype stack
  = Bot
  | $ of stack * stack_item

infix 2 $

fun pp_item (Tok(t,r)) = "Tok(" ^ T.toString t ^ ")"
  | pp_item (Ids _) = "Ids"
  | pp_item (TpInfix _) = "TpInfix"
  | pp_item (TpAlts _) = "TpAlts"
  | pp_item (Tp _) = "Tp"
  | pp_item (Parms _) = "Parms"
  | pp_item (Cmd _) = "Exp"
  | pp_item (Pat _) = "Pat"
  | pp_item (Branches _) = "Branches"
  | pp_item (Defn _) = "Defn"
  | pp_item (Failure _) = "Failure"
  | pp_item (Value _) = "Value"
  | pp_item (ValDefn _) = "ValDefn"
  | pp_item (Error) = "Error"

fun pp_stack Bot = "Bot"
  | pp_stack (S $ item) = pp_stack S ^ " $ " ^ pp_item item

(* This is a hand-written shift/reduce parser
 * I have tried to resist the temptation to optimize or transform,
 * since it is very easy to make mistakes.
 *
 * Parsing functions are named p_<nonterminal>, possibly with a suffix
 * for intermediate parsing states.  Reducing functions are named
 * r_<nonterminal>, possibly with a suffix for intermediate states.
 * With few exceptions, parsing functions have type
 *
 * p_<nonterminal> : stack * L.lexresult M.Front -> stack * L.lexresult M.Front
 * r_<nonterminal> : stack -> stack
 *
 * Note that in input and output of the parsing function, the first
 * token of the lexer stream is exposed (type L.lexresult M.Front) which
 * make for easy matching and composition of these functions.
 *
 * Generally p_<nt> will consume terminals for an <nt> from the lexresult
 * stream and return with the resulting abstract syntax for <nt> on the stack.
 * Generally r_<nt> will consume a mix of terminal and nonterminals on
 * the stack and push the abstract syntax for an <nt> onto the stack.
 *
 * While parsing expression with infix, prefix, and postfix operators
 * we continue parsing, extending the expression until we encounter
 * a terminal that completes that is not part of an expression.
 *
 * p_<nt> is a function that parses nonterminal <nt>
 * r_<nt> is a function that reduces nonterminal <nt>
 * m_<nt> is a function that marks nonterminal <nt> with region information
 *)

(***********************)
(* Parsing Combinators *)
(***********************)

(* Always call 'first ST' to extract the first token for examination *)
fun first (S, M.Cons((t, r), ts')) = t
fun second (S, ft) = ft

fun shift (S, M.Cons((t, r), ts')) = (S $ Tok(t, r), M.force ts')
fun reduce reduce_fun (S, ft) = (reduce_fun S, ft)

fun drop (S, M.Cons((t, r), ts')) = (S, M.force ts') (* use sparingly *)
fun push item (S, ft) = (S $ item, ft)

fun >>(f,g) = fn x => g(f(x))
fun |>(x,f) = f x

infixr 2 >>
infix 1 |>

(* region manipulation *)
fun join (left1, right1) (left2, right2) = (left1, right2)
fun here (S, M.Cons((t, r), ts')) = r
fun last (Bot) = 0
  | last (S $ Tok(t,(l,r))) = r
  | last (S $ Ids(_,(l,r))) = r
  | last (S $ TpInfix(_,_,(l,r))) = r
  | last (S $ TpAlts(_,(l,r))) = r
  | last (S $ Tp(_,(l,r))) = r
  | last (S $ Parms(_,(l,r))) = r
  | last (S $ Cmd(_,(l,r))) = r
  | last (S $ Pat(_,(l,r))) = r
  | last (S $ Branches(_,(l,r))) = r
  | last (S $ Defn(_,(l,r))) = r
  | last (S $ Failure(l,r)) = r
  | last (S $ Value(_, (l, r))) = r
  | last (S $ ValDefn(_, (l, r))) = r
  | last (S $ Error) = 0 (* should not be asked *)
val nowhere = (0,0)

(****************************)
(* Building abstract syntax *)
(****************************)

fun m_tp (tau, (left, right)) = A.MarkedTp (Mark.mark' (tau, PS.ext (left, right)))

fun m_cmd (P, (left, right)) = A.MarkedCmd (Mark.mark' (P, PS.ext (left, right)))

fun m_pat (pat, (left, right)) = A.MarkedPat (Mark.mark' (pat, PS.ext (left, right)))

fun m_value (value, (left, right)) = A.MarkedValue (Mark.mark' (value, PS.ext (left, right)))

(***********)
(* Parsing *)
(***********)

(*
 * Refer to the grammar in readme.txt
 * Comments refer to the nonterminals shown there in <angle brackets>
 *)

(* <defn> *)
fun p_defn ST = case first ST of
    T.TYPE => ST |> shift >> p_id >> p_terminal T.EQ >> p_tp >> reduce r_defn
  | T.PROC => ST |> shift >> p_id >> p_parms >> p_terminal T.EQ >> p_cmd >> reduce r_defn
  | T.FAIL => ST |> shift >> reduce r_defn
  | T.EOF => ST
  | t => parse_error (here ST, "unexpected token " ^ pp_tok t ^ " at top level") (second ST)

and r_defn (S $ Tok(T.TYPE,r1) $ Tok(T.IDENT(a),_) $ Tok(T.EQ,_) $ Tp(tau,r2)) =
    S $ Defn(A.TypeDefn(a, tau, PS.ext(join r1 r2)), join r1 r2)
  | r_defn (S $ Tok(T.PROC,r1) $ Tok(T.IDENT(p),_) $ Parms(parm::parms, _) $ Tok(T.EQ,_) $ Cmd(P,r2)) =
    S $ Defn(A.ProcDefn(p, parm, parms, P, PS.ext(join r1 r2)), join r1 r2)
  | r_defn (S $ Tok(T.FAIL,r)) = S $ Failure(r)
  | r_defn S = raise Match (* TextIO.print (pp_stack S ^ "\n") *)

(* <parm>+ *)
and p_parms ST = case first ST of
    T.LPAREN => ST |> push (Parms(nil, here ST)) >> p_parm_seq (* keep LPAREN *)
  | t => parse_error (here ST, "each procedure must have a destination") (second ST)

and p_parm_seq ST = case first ST of
    T.LPAREN => ST |> shift >> p_id >> p_terminal T.COLON >> p_tp >> p_terminal T.RPAREN
                   >> reduce r_parm >> p_parm_seq
  | _ => ST

and r_parm (S $ Parms(xtaus, r1) $ Tok(T.LPAREN, _) $ Tok(T.IDENT(x), _) $ Tok(T.COLON, _)
              $ Tp(tau, _) $ Tok(T.RPAREN,r2)) =
    S $ Parms(xtaus @ [(x,tau)], join r1 r2)

(* <tp> *)
and p_tp ST = case first ST of
    T.IDENT(a) => ST |> shift >> reduce r_atom_tp >> p_tp_prec
  | T.NAT(1) => ST |> shift >> reduce r_atom_tp >> p_tp_prec
  | T.NAT(n) => parse_error (here ST, "invalid natural number (only '1' is a type)") (second ST)
  | T.STAR => ST |> drop >> push (TpInfix(2, A.Times, here ST)) >> p_tp_prec
  | T.PLUS => ST |> shift >> p_terminal T.LBRACE >> push (TpAlts(nil, here ST))
                 >> p_alts >> p_terminal T.RBRACE >> reduce r_atom_tp >> p_tp_prec
  | T.LPAREN => ST |> shift >> p_tp >> p_terminal T.RPAREN >> reduce r_atom_tp >> p_tp
  | _ => ST |> reduce (r_tp ST) (* type complete: reduce *)

and p_tp_prec (ST as (S,ft)) = case S of
    S $ Tp(tau1,r1) $ Tp(tau2,r2) => parse_error (join r1 r2, "consecutive types") ft
  | S $ Tp _ => p_tp ST (* tp leading, after infix or at beginning *)

  | S $ Tp(tau1,r1) $ TpInfix(p, f, _) $ Tp(tau2, r2) $ TpInfix(p', f', r') =>
    if p > p' then p_tp_prec (S $ Tp(m_tp(f(tau1,tau2), join r1 r2), join r1 r2) $ TpInfix(p', f', r'), ft)
                   (* reduce, right assoc *)
    else p_tp ST (* shift *)
  | S $ Tp _ $ TpInfix _ => p_tp ST (* shift *)
  (* errors *)
  | S $ TpInfix(_,_,r) $ TpInfix(_,_,r') => parse_error (join r r', "consecutive infix operators") ft
  | S $ TpInfix(_,_,r) => parse_error (r, "leading infix operator") ft
  | S => raise Match (* seems like these are all possibilities? *)

(* <alts> *)
and p_alts (ST as (S, ft)) = case first ST of
    T.LABEL(l) => ST |> shift >> p_terminal T.COLON >> p_tp >> reduce r_alts >> p_alts_next
  | t as T.IDENT(id) => error_expected_h
                        (here ST, T.IDENT("<label>"), t, "forgot ' to start tag?") ft
  | t => error_expected (here ST, T.IDENT("<label>"), t) ft

(* [ ',' <alts> ] *)
and p_alts_next ST = case first ST of
    T.COMMA => ST |> drop >> p_alts
  | _ => ST

(* reduce <alts> *)
and r_alts (S $ TpAlts(alts, r1) $ Tok(T.LABEL(l), _) $ Tok(T.COLON, _) $ Tp(tau, r2)) =
    S $ TpAlts(alts @ [(l,tau)], join r1 r2)
  | r_alts S = raise Match

and r_atom_tp (S $ Tok(T.IDENT(a),r)) = S $ Tp(m_tp(A.TpName(a), r), r)
  | r_atom_tp (S $ Tok(T.NAT(1),r)) = S $ Tp(m_tp(A.One, r),r)
  | r_atom_tp (S $ Tok(T.LPAREN,r1) $ Tp(tau,_) $ Tok(T.RPAREN,r2)) = S $ Tp(tau,join r1 r2)
  | r_atom_tp (S $ Tok(T.PLUS,r1) $ Tok(T.LBRACE,_) $ TpAlts(alts,_) $ Tok(T.RBRACE,r2)) =
    S $ Tp(m_tp(A.Plus(alts), join r1 r2), join r1 r2)

(* reduce <tp> *)
and r_tp ST (S $ Tp(tau1,r1) $ TpInfix(p,f,_) $ Tp(tau2,r2)) =
    r_tp ST (S $ Tp(m_tp(f(tau1,tau2), join r1 r2), join r1 r2))
  | r_tp ST (S $ Tp(tau1,r1) $ Tp(tau2,r2)) = parse_error (join r1 r2, "consecutive types") (second ST)
  | r_tp ST (S $ Tp(tau,r)) = S $ Tp(tau,r)  (* must come after prefix, infix, and juxtaposition cases *)
  | r_tp ST (S $ TpInfix(p,f,r)) = parse_error (join r (here ST), "incomplete type") (second ST)
  | r_tp ST S = parse_error (here ST, "empty type") (second ST)
    
(* <cmd> *)
and p_cmd (ST as (S, ft)) = case first ST of
    T.READ => ST |> shift >> p_id >> p_branch1 >> reduce r_cmd
  | T.WRITE => ST |> shift >> p_id >> p_pat >> reduce r_cmd
  | T.CUT => ST |> shift >> p_id >> p_terminal T.COLON >> p_tp >> p_cmd >> p_cmd >> reduce r_cmd
  | T.ID => ST |> shift >> p_id >> p_id >> reduce r_cmd
  | T.CALL => ST |> shift >> p_id >> p_ids1 >> reduce r_cmd
  | T.LBRACE => ST |> shift >> p_cmd >> p_terminal T.RBRACE >> reduce r_cmd
  | t as T.IDENT(p) => error_expected_list_h
                           (here ST, [T.READ, T.WRITE, T.CUT, T.ID, T.CALL], t, "forgot 'call'?") ft
  | t => error_expected_list (here ST, [T.READ, T.WRITE, T.CUT, T.ID, T.CALL], t) ft

and r_cmd (S $ Tok(T.READ, r1) $ Tok(T.IDENT(x), _) $ Branches(branches, r2)) =
    S $ Cmd(m_cmd(A.Read(x, branches), join r1 r2), join r1 r2)
  | r_cmd (S $ Tok(T.WRITE, r1) $ Tok(T.IDENT(x), _) $ Pat(v, r2)) =
    S $ Cmd(m_cmd(A.Write(x, v), join r1 r2), join r1 r2)
  | r_cmd (S $ Tok(T.CUT, r1) $ Tok(T.IDENT(x), _) $ Tok(T.COLON, _) $ Tp(tau, _) $ Cmd(P, r2) $ Cmd(Q, r3)) =
    S $ Cmd(m_cmd(A.Cut(x, tau, P, Q), join r1 r2), join r1 r2)
  | r_cmd (S $ Tok(T.ID, r1) $ Tok(T.IDENT(x), _) $ Tok(T.IDENT(y), r2)) =
    S $ Cmd(m_cmd(A.Id(x, y), join r1 r2), join r1 r2)
  | r_cmd (S $ Tok(T.CALL, r1) $ Tok(T.IDENT(p), _) $ Ids(x::ys, r2)) =
    S $ Cmd(m_cmd(A.Call(p, x, ys), join r1 r2), join r1 r2)
  | r_cmd (S $ Tok(T.LBRACE, r1) $ Cmd(P, _) $ Tok(T.RBRACE, r2)) =
    S $ Cmd(m_cmd(P, join r1 r2), join r1 r2)
  | r_cmd S = raise Match (* ( TextIO.print (pp_stack S ^ "\n") ; raise Match ) *)

(* <pat> <cmd> | '{' <branch>+ '}' *)
and p_branch1 (ST as (S, ft)) = case first ST of
    T.LBRACE => ST |> shift >> push (Branches(nil, here ST)) >> p_branches >> p_terminal T.RBRACE
                   >> reduce r_branch1
  | T.BAR => parse_error (here ST, "expected '{' or <pat>, found '|'" ^^ "enclose branches in braces?") ft
  | _ => ST |> p_pat >> p_cmd >> reduce r_branch1

and r_branch1 (S $ Pat(pat, r1) $ Cmd(P, r2)) =
    S $ Branches([(pat, P)], join r1 r2)
  | r_branch1 (S $ Tok(T.LBRACE, r1) $ Branches(branches, _) $ Tok(T.RBRACE, r2)) =
    S $ Branches(branches, join r1 r2)

(* <branch>* *)
and p_branches ST = case first ST of
    T.BAR => ST |> drop >> p_pat >> p_terminal T.RIGHTARROW >> p_cmd >> reduce r_branch >> p_branches
  | _ => ST

and r_branch (S $ Branches(branches,r1) $ Pat(pat,_) $ Tok(T.RIGHTARROW,_) $ Cmd(P, r2)) =
    S $ Branches(branches @ [(pat, P)], join r1 r2)

(* <pat> *)
and p_pat ST = case first ST of
    T.LPAREN => ST |> shift >> p_pat_opt >> p_terminal T.RPAREN >> reduce r_pat
  | T.LABEL(l) => ST |> shift >> p_terminal T.LPAREN >> p_id >> p_terminal T.RPAREN >> reduce r_pat
  | T.IDENT(x) => error_expected_list_h (here ST, [T.LPAREN, T.IDENT("<label>")], T.IDENT(x),
                                         "add single quote before label?")
                                        (second ST)
  | t => error_expected_list (here ST, [T.LPAREN, T.IDENT("<label>")], t) (second ST)

and p_pat_opt ST = case first ST of
    T.RPAREN => ST              (*  "()" for the unit *)
  | _ => ST |> p_id >> p_terminal T.COMMA >> p_id

and r_pat (S $ Tok(T.LPAREN,r1) $ Tok(T.RPAREN,r2)) = S $ Pat(m_pat(A.UnitPat,join r1 r2),join r1 r2)
  | r_pat (S $ Tok(T.LPAREN,r1) $ Tok(T.IDENT(x), _) $ Tok(T.COMMA, _) $ Tok(T.IDENT(y), _) $ Tok(T.RPAREN, r2))
    = S $ Pat(m_pat(A.PairPat(x,y), join r1 r2), join r1 r2)
  | r_pat (S $ Tok(T.LABEL(l), r1) $ Tok(T.LPAREN, _) $ Tok(T.IDENT(x), _) $ Tok(T.RPAREN, r2))
    = S $ Pat(m_pat(A.InjPat(l,x), join r1 r2), join r1 r2)
    
(* <id> *)
and p_id ST = case first ST of
    T.IDENT(id) => ST |> drop >> push (Tok(T.IDENT(id), here ST))
  | t => parse_error (here ST, "expected identifier, found " ^ pp_tok t) (second ST)

(* <id>+ *)
and p_ids1 ST = case first ST of
    T.IDENT(id) => ST |> p_ids
  | t => parse_error (here ST, "call requires a destination") (second ST)

(* <id>* *)
and p_ids ST = ST |> push (Ids(nil, here ST)) >> p_id_seq

and p_id_seq ST = case first ST of         
    T.IDENT(id) => ST |> drop >> push (Tok(T.IDENT(id), here ST)) >> reduce r_ids >> p_id_seq
  | _ => ST

and r_ids (S $ Ids(ids, r1) $ Tok(T.IDENT(id),r2)) = S $ Ids(ids @ [id], join r1 r2)

(* <label> *)
and p_label ST = case first ST of
    T.LABEL(i) => ST |> shift
  | t => parse_error (here ST, "expected <label>, found " ^ pp_tok t) (second ST)

(* parse any token (terminal symbol) 't_needed' *)
and p_terminal t_needed ST = case first ST of
    t => if t_needed = t
	 then ST |> shift
	 else error_expected (here ST, t_needed, t) (second ST)

fun p_value ST = case first ST of
    T.LPAREN => ST |> shift >> p_value_opt >> p_terminal T.RPAREN >> reduce r_value
  | T.LABEL(l) => ST |> shift >> p_value >> reduce r_value
  | t => error_expected_list (here ST, [T.LPAREN, T.IDENT("<label>")], t) (second ST)

and p_value_opt ST = case first ST of
    T.RPAREN => ST  (* "()" for the unit value *)
  | _ => ST |> p_value >> p_terminal T.COMMA >> p_value

and r_value (S $ Tok(T.LPAREN, r1) $ Tok(T.RPAREN, r2)) = S $ Value(m_value(A.Unit, join r1 r2), join r1 r2)
  | r_value (S $ Tok(T.LPAREN, r1) $ Value(V1, _) $ Tok(T.COMMA, _) $ Value(V2, _) $ Tok(T.RPAREN, r2)) =
    S $ Value(m_value(A.Pair(V1, V2), join r1 r2), join r1 r2)
  | r_value (S $ Tok(T.LABEL(l), r1) $ Value(V, r2)) = S $ Value(m_value(A.Inj(l, V), join r1 r2), join r1 r2)

fun p_valdefn ST = case first ST of
    T.VALUE => ST |> shift >> p_id >> p_terminal T.EQ >> p_value >> reduce r_valdefn
  | T.EOF => ST
  | t => parse_error (here ST, "unexpected token " ^ pp_tok t ^ " at top level") (second ST)

and r_valdefn (S $ Tok(T.VALUE, r1) $ Tok(T.IDENT(p), _) $ Tok(T.EQ, _) $ Value(V, r2)) =
    S $ ValDefn(A.ValDefn(p, V, PS.ext(join r1 r2)), join r1 r2)

(* recover from a parsing error to continue *)
(* this skips ahead to the next top level declaration keyword *)
fun recover (tf as M.Nil) = (Bot $ Error, tf)
  | recover (tf as M.Cons((t,r), ts)) =
    ( case t
       of T.TYPE => (Bot $ Error, tf)
        | T.PROC => (Bot $ Error, tf) (* token stream including t *)
        | T.FAIL => (Bot $ Error, tf)
        | T.EOF => (Bot $ Error, tf)
        | _ => (* skip all other tokens *)
          recover' ts )
and recover' ts = recover (next_valid ts)

and next_valid ts = M.force ts handle Lex.LexError(ts') => next_valid ts'

(* robustly parsing a definition intercepts lexing and
 * parsing errors with a continuation stream and then
 * skips ahead to the next top-level keyword
 *)
fun robust_p_defn ST =
    p_defn ST
    handle Lex.LexError ts => recover' ts
         | ParseError tf => recover tf

(* <defns>* *)
(* this allows syntax errors to be negated *)
fun parse_defns_posneg negated token_front =
    let 
        val ST = if negated then ErrorMsg.suppress (fn () => robust_p_defn (Bot, token_front))
                 else robust_p_defn (Bot, token_front)
    in
        case ST
         of (Bot $ Error, token_front) => (* error in preceding definition *)
            if negated then ( if !Flags.verbosity >= 3
                              then TextIO.print "% ignored expected syntax error\n"
                              else ()
                            ; parse_defns_posneg false token_front )
            else raise ErrorMsg.Error
          | (Bot $ Failure(r), token_front) =>
            if negated then ( ErrorMsg.error (PS.ext r) "double negation"
                            ; raise ErrorMsg.Error  )
            else parse_defns_posneg true token_front
          | (Bot, M.Cons((T.EOF, r), token_front)) => (* whole file processed *)
            if negated then ( ErrorMsg.error (PS.ext r) "negation not followed by definition"
                            ; raise ErrorMsg.Error )
            else []
          | (Bot $ Defn(defn,r), token_front) =>
            (if negated then A.FailDefn(defn, PS.ext r) else defn) :: parse_defns_posneg false token_front
          (* should be the only possibilities *)
    end

fun negate1 r (defn :: defns) = A.FailDefn(defn, PS.ext r)::defns
  | negate1 r nil = ( ErrorMsg.error (PS.ext r) "negation not followed by definition"
                    ; nil )
(* <defns>* *)
(* without negation of syntax errors *)
fun parse_defns token_front =
    case robust_p_defn (Bot, token_front)
     of (Bot $ Error, token_front) =>
        (* ignore error item, if there is one *)
        (* error is implicitly recorded in ErrorMsg.anyErrors *)
        parse_defns token_front
      | (Bot $ Failure(r), token_front) =>
        negate1 r (parse_defns token_front)
      | (Bot $ Defn(defn, r), token_front) =>
        defn :: parse_defns token_front
      | (Bot, M.Cons((T.EOF, r), token_front)) => (* whole file processed *)
        nil

(* parse_sax filename = defns
 * first apply lexer, the parser to the resulting token stream
 * raise ErrorMsg.Error if not lexically or syntactically correct
 * or if file does not exist or cannot be opened
 *)
fun parse_sax filename =
    SafeIO.withOpenIn filename (fn instream =>
      let val () = PS.pushfile filename (* start at pos 0 in filename *)
          val token_stream = Lex.makeLexer (fn _ => TextIO.input instream)
          val defns = parse_defns (M.force token_stream)
          val () = if !ErrorMsg.anyErrors then raise ErrorMsg.Error else ()
          val () = PS.popfile ()
      in defns end)
    handle e as IO.Io _ => ( ErrorMsg.error NONE (exnMessage e)
                           ; raise ErrorMsg.Error )

fun robust_p_valdefn ST =
    p_valdefn ST
    handle Lex.LexError ts => recover' ts
         | ParseError tf => recover tf

fun parse_valdefns token_front =
    case robust_p_valdefn (Bot, token_front)
     of (Bot $ Error, token_front) => raise ErrorMsg.Error
      | (Bot, M.Cons((T.EOF, r), token_front)) => (* whole file processed *)
        []
      | (Bot $ ValDefn(valdefn, r), token_front) =>
        valdefn :: parse_valdefns token_front

fun parse_val filename =
    SafeIO.withOpenIn filename (fn instream =>
      let val () = PS.pushfile filename
          val token_stream = Lex.makeLexer (fn _ => TextIO.input instream)
          val valdefns = parse_valdefns (M.force token_stream)
          val () = PS.popfile ()
      in
          valdefns
      end)
    handle e as IO.Io _ => ( ErrorMsg.error NONE (exnMessage e)
                           ; raise ErrorMsg.Error )

end (* structure Parse *)
