(* External Syntax *)

(* Types and command are all in one structure, since
 * external syntax is only used for parsing purposes
 *
 * External syntax does not distinguish between global (contextually
 * typed) variables and ordinary bound variables.  This is
 * resolved in elaboration.
 *)

signature AST =
sig

    type label = string
    type tpname = string

    datatype tp = Times of tp * tp
                | One
                | Plus of (label * tp) list
                | TpName of tpname
                | MarkedTp of tp Mark.marked

    type varname = string
    type procname = string

    datatype pat = PairPat of varname * varname
                 | UnitPat
                 | InjPat of label * varname
                 | MarkedPat of pat Mark.marked

    datatype cmd = Read of varname * (pat * cmd) list
                 | Write of varname * pat
                 | Cut of varname * tp * cmd * cmd
                 | Id of varname * varname
                 | Call of procname * varname * varname list
                 | MarkedCmd of cmd Mark.marked

    type parm = varname * tp

    type ext = Mark.ext option

    datatype defn = TypeDefn of tpname * tp * ext
                  | ProcDefn of procname * parm * parm list * cmd * ext
                  | FailDefn of defn * ext

    type env = defn list

    (* these print redundant '(...)' and '{...}' *)
    (* see PPSax for prettier output *)
    val pp_tp : tp -> string
    val pp_pat : pat -> string
    val pp_cmd : int -> cmd -> string 
    val pp_defn : defn -> string
    val pp_env : env -> string

    datatype value = Pair of value * value
                   | Unit
                   | Inj of label * value
                   | MarkedValue of value Mark.marked

    datatype valdefn = ValDefn of procname * value * ext

    type valenv = valdefn list

end (* signature AST *)

structure Ast :> AST =
struct

    type label = string
    type tpname = string

    datatype tp = Times of tp * tp
                | One
                | Plus of (label * tp) list
                | TpName of tpname
                | MarkedTp of tp Mark.marked

    type varname = string
    type procname = string

    datatype pat = PairPat of varname * varname
                 | UnitPat
                 | InjPat of label * varname
                 | MarkedPat of pat Mark.marked

    datatype cmd = Read of varname * (pat * cmd) list
                 | Write of varname * pat
                 | Cut of varname * tp * cmd * cmd
                 | Id of varname * varname
                 | Call of procname * varname * varname list
                 | MarkedCmd of cmd Mark.marked

    type parm = varname * tp

    type ext = Mark.ext option

    datatype defn = TypeDefn of tpname * tp * ext
                  | ProcDefn of procname * parm * parm list * cmd * ext
                  | FailDefn of defn * ext

    type env = defn list

    fun indent 0 s = s
      | indent n s = " " ^ indent (n-1) s

    fun pp_label l = "'" ^ l

    fun parens s = "(" ^ s ^ ")"

    fun pp_tp (Times(A,B)) = parens (pp_tp A ^ " * " ^ pp_tp B)
      | pp_tp (One) = "1"
      | pp_tp (Plus(alts)) =
        "+" ^ "{" ^ String.concatWith ", " (List.map (fn (l, Al) => pp_label l ^ " : " ^ pp_tp Al) alts) ^ "}"
      | pp_tp (TpName(a)) = a
      | pp_tp (MarkedTp mA) = pp_tp (Mark.data mA)

    fun pp_pat (PairPat(x,y)) = "(" ^ x ^ ", " ^ y ^ ")"
      | pp_pat (UnitPat) = "()"
      | pp_pat (InjPat(l,x)) = pp_label l ^ "(" ^ x ^ ")"
      | pp_pat (MarkedPat mpat) = pp_pat (Mark.data mpat)

    fun pp_cmd col (Read(x, [(pat,P)])) =
        "read " ^ x ^ " " ^ pp_pat pat ^ "\n"
        ^ indent col (pp_cmd col P)
      | pp_cmd col (Read(x, branches)) =
        (* in a branch; use braces for scope *)
        "read " ^ x ^ " {\n"
        ^ pp_branches col branches
        ^ "\n" ^ indent col "}"
      | pp_cmd col (Write(x, pat)) =
        "write " ^ x ^ " " ^ pp_pat pat
      | pp_cmd col (Cut(x, A, P, Q)) =
        "cut " ^ x ^ " : " ^ pp_tp A ^ " {\n"
        ^ indent (col+4) (pp_cmd (col+4) P ^ "\n")
        ^ indent col "}\n"
        ^ indent col (pp_cmd col Q)
      | pp_cmd col (Id(x, y)) =
        "id " ^ x ^ " " ^ y
      | pp_cmd col (Call(p, x, ys)) =
        "call " ^ p ^ " " ^ x ^ " " ^ String.concatWith " " ys
      | pp_cmd col (MarkedCmd mP) = pp_cmd col (Mark.data mP)

    and pp_branches col ((pat, P)::nil) = pp_branch col (pat, P)
      | pp_branches col ((pat, P)::branches) =
        pp_branch col (pat, P) ^ "\n" ^ pp_branches col branches
      | pp_branches col (nil) = raise Match

    and pp_branch col (pat, P) =
        let val prefix = "| " ^ pp_pat pat ^ " => "
            val k = String.size prefix
        in
            indent col (prefix ^ pp_cmd (col + k) P)
        end

    fun pp_parm (x, A) = parens (x ^ " : " ^ pp_tp A)

    fun pp_parms yBs = String.concatWith " " (List.map pp_parm yBs)

    fun pp_defn (TypeDefn(a, A, _)) = "type " ^ a ^ " = " ^ pp_tp A ^ "\n"
      | pp_defn (ProcDefn(p, xA, yBs, P, _)) =
        "proc " ^ p ^ " " ^ pp_parm xA ^ " " ^ pp_parms yBs ^ " =\n"
        ^ indent 4 (pp_cmd 4 P) ^ "\n"
      | pp_defn (FailDefn(defn, _)) = "fail\n" ^ pp_defn defn

    fun pp_env defns = String.concatWith "\n" (List.map pp_defn defns)


    datatype value = Pair of value * value
                   | Unit
                   | Inj of label * value
                   | MarkedValue of value Mark.marked

    datatype valdefn = ValDefn of procname * value * ext

    type valenv = valdefn list

end (* structure Ast *)
