signature TOP =
sig
    (* interactive loop *)
    val loop_print : unit -> unit
    val loop_print_noDB : unit -> unit  (* print before translating to deBruijn form *)
    val loop_type  : unit -> unit
    val loop_eval  : unit -> unit
    val loop_step  : unit -> unit

    (* read a MinML source file *)
    val file_print : string -> unit
    val file_print_noDB : string -> unit (* print before translating to deBruijn form *)
    val file_type  : string -> unit
    val file_eval  : string -> unit
    val file_step  : string -> unit
(* new asst8 code: *)
    val apply_eval  : string -> DBMinML.exp -> unit
    val apply_step  : string -> DBMinML.exp -> unit

    val ordered_random : (int * int) ->  int -> DBMinML.exp
       (* ordered_random (seed1,seed2) n:
           Return an ordered tree of up to n random integers in 
           the range 1..n, no duplicates *)

    val random_seq : (int * int) -> int -> DBMinML.exp
       (* random_seq (seed1,seed2) n:
           Return a rightist tree of n random integers in 
           the range 1..n.  The result may have duplicates. *)

    val seq : int -> DBMinML.exp        (* Return a rightist tree *)
    val complete : int -> DBMinML.exp   (* Return a complete n-level tree *)
(* end asst8 code *)
end

structure Top :> TOP =
struct

    fun loop_print () =
        (Loop.loop Loop.show)
        handle Parse.Error s => print ("Parse Error: " ^ s ^ "\n");

    fun loop_print_noDB () =
        (Loop.loop_noDB Loop.show_noDB)
        handle Parse.Error s => print ("Parse Error: " ^ s ^ "\n");

    fun loop_type () = 
        (Loop.loop Loop.showType)
        handle Parse.Error s => print ("Parse Error: " ^ s ^ "\n");

    fun loop_eval () = 
        (Loop.loop (Loop.eval Loop.showType)) 
        handle Parse.Error s => print ("Parse Error: " ^ s ^ "\n");

    fun loop_step () =
        (Loop.loop_state (Loop.step Loop.showState))
        handle Parse.Error s => print ("Parse Error: " ^ s ^ "\n");

    fun file_print f =
        (Loop.loopFile f Loop.show)
        handle Parse.Error s => print ("Parse Error: " ^ s ^ "\n");

    fun file_print_noDB f =
        (Loop.loopFile_noDB f Loop.show_noDB)
        handle Parse.Error s => print ("Parse Error: " ^ s ^ "\n");

    fun file_type  f = 
        (Loop.loopFile f Loop.showType)
        handle Parse.Error s => print ("Parse Error: " ^ s ^ "\n");

    fun file_eval  f =
        (Loop.loopFile f (Loop.eval Loop.showType)) 
        handle Parse.Error s => print ("Parse Error: " ^ s ^ "\n");

    fun file_step  f =
        (Loop.loopFile_state f (Loop.step Loop.showState))
        handle Parse.Error s => print ("Parse Error: " ^ s ^ "\n");
(* new asst8 code: *)
    fun apply_eval f arg =
        let val e = Stream.hd ((Translate.translate o Parse.parse o 
                                Lexer.lex) (Input.readfile f)) 
        in
          case EMach.multiStep (EMach.start arg)
           of EMach.Done (v) =>
              (case EMach.evaluate (EMach.start e) v
                of EMach.Done (v') =>
                   print (DBPrint.expToString (case v'
                                                of EMach.VClosure(_, f) => f
                                                 | _ => EMach.expOf v')
                          ^ "\n")
                 | _ => print "Unexpected EMach result!\n")
            | _ => print "Couldn't evaluate argument!\n"
        end
        handle Parse.Error s => print ("Parse error: " ^ s ^ "\n");

    fun apply_step f arg =
        let val e = Stream.hd ((Translate.translate o Parse.parse o 
                                Lexer.lex) (Input.readfile f)) 
        in
          case EMach.multiStep (EMach.start arg)
           of EMach.Done (v) =>
              Stream.app Loop.showState 
                         (EMach.evaluateStream (EMach.start e) v)
            | _ => print "Couldn't evaluate argument!\n"
        end
        handle Parse.Error s => print ("Parse error: " ^ s ^ "\n");

    open DBMinML

    fun seq' f n =
	if n <= 0 then Leaf else Node(f n, Leaf, seq' f (n - 1))

    fun complete' f n =
	if n <= 0 then Leaf
	else
	    let val subtree = complete' f (n-1)
	    in
		Node(f n, subtree, subtree)
	    end

    val seq = seq' (fn n => Int n)
    val complete = complete' (fn n => Int n)

    fun random_seq seed n =
       let val r = Random.rand seed
       in
	   seq' (fn _ => Int(Random.randRange(1, n) r)) n
       end

    fun insert k exp =
	case exp of
	    Leaf => Node(Int(k), Leaf, Leaf)
	  | Node(Int(x),l,r) =>
		if k = x then Node(Int(x), l, r)
		else if k < x then Node(Int(x), insert k l, r)
		     else Node(Int(x), l, insert k r)
          | _ => raise Match (* silence warning *)

    fun ordered_random seed n =
	let val r = Random.rand seed
	    fun aux n =
		if n <= 0 then Leaf
		else
		    insert (Random.randRange(1, n) r) (aux (n-1))
	in
	    aux n
	end
(* end asst8 code *)
end
