signature MACH =
sig

  exception Stuck

  type exp

  datatype binding =
      BOrd of value             (* ordinary, non-recursive bindings *)
    | BRec of value             (* Recusive bindings i.e. =*= *)

  and value =
      VInt of int
    | VBool of bool
    | VUnit
    | VPair of value * value
    | VClosure of binding list * exp
    | VLeaf
    | VNode of value * value * value

  and frame = 
      FPrimopN of P.primop * value list * exp list
    | FIf of exp * exp          (* if(_, e1, e2) *)
    | FPair1 of exp             (* pair(_, e2) *)
    | FPair2 of value           (* pair(v1,_) *)
    | FFst                      (* first(_) *)
    | FSnd                      (* snd(_) *)
    | FApply1 of exp            (* apply(_, e2) *)
    | FApply2 of value          (* apply(v1, _) *)
    | FLet of exp               (* let(_, e2) *)
    | FNode1 of exp * exp       (* Node(_, el, er) *)
    | FNode2 of value * exp     (* Node(v, _, er) *)
    | FNode3 of value * value   (* Node(v, vl, _) *)
    | FCase of exp * exp        (* case(_, e1, x.l.r.e2) *)
    | FEnv of binding list

  type stack

  datatype state = 
      Eval of stack * binding list * exp
    | Return of stack * binding list * value
    | Done of value
  (* TODO extend this datatype according to your rules *)

  (* partial function from machine values to expressions *)
  val expOf : value -> exp

  (* partial function from expressions to machine values *)
  val valueOf : exp -> value

  (* returns a new state with an empty stack and an empty environment *)
  val start : exp -> state

  (* one-step evaluation, raises Stuck if impossible *)
  val step : state -> state

  (* repeats step as many times as possible *)
  val multiStep : state -> state

  (* a stream of all steps of an evaluation *)
  val stepStream : state -> state Stream.stream

  (* evaluates the state as a function with the given argument *)
  val evaluate : state -> value -> state

  (* a stream of all steps for a function with the given argument *)
  val evaluateStream : state -> value -> state Stream.stream

end;  (* signature MACH *)

structure EMach : MACH =
struct

  open T
  open P
  structure DB = DBMinML
  structure S = Stream

  exception Stuck

  type exp = DB.exp

  datatype binding =
      BOrd of value             (* ordinary, non-recursive bindings *)
    | BRec of value             (* Recusive bindings i.e. =*= *)

  and value =
      VInt of int
    | VBool of bool
    | VUnit
    | VPair of value * value
    | VClosure of binding list * exp
    | VLeaf
    | VNode of value * value * value

  and frame = 
      FPrimopN of primop * value list * exp list
    | FIf of exp * exp          (* if(_, e1, e2) *)
    | FPair1 of exp             (* pair(_, e2) *)
    | FPair2 of value           (* pair(v1,_) *)
    | FFst                      (* first(_) *)
    | FSnd                      (* snd(_) *)
    | FApply1 of exp            (* apply(_, e2) *)
    | FApply2 of value          (* apply(v1, _) *)
    | FLet of exp               (* let(_, e2) *)
    | FNode1 of exp * exp       (* Node(_, el, er) *)
    | FNode2 of value * exp     (* Node(v, _, er) *)
    | FNode3 of value * value   (* Node(v, vl, _) *)
    | FCase of exp * exp        (* case(_, e1, x.l.r.e2) *)
    | FEnv of binding list

  type stack = frame list

  datatype state = 
      Eval of stack * binding list * exp
    | Return of stack * binding list * value
    | Done of value
  (* TODO extend this datatype according to your rules *)

  fun expOf (VInt i) = DB.Int (i)
    | expOf (VBool b) = DB.Bool (b)
    | expOf (VUnit) = DB.Unit
    | expOf (VPair (v1, v2)) = DB.Pair (expOf v1, expOf v2)
    | expOf (VLeaf) = DB.Leaf
    | expOf (VNode (x, l, r)) = DB.Node (expOf x, expOf l, expOf r)
    | expOf _ = raise Match

  fun valueOf (DB.Int (i)) = VInt (i)
    | valueOf (DB.Bool (b)) = VBool (b)
    | valueOf (DB.Unit) = VUnit
    | valueOf (DB.Pair (e1, e2)) = VPair (valueOf e1, valueOf e2)
    | valueOf (DB.Leaf) = VLeaf
    | valueOf (DB.Node (x, l, r)) = VNode (valueOf x, valueOf l, valueOf r)
    | valueOf _ = raise Match

  fun start e = Eval (nil, nil, e)

  (* maintain state about previous process identifiers *)
  val nextPID = ref 0
  fun getNextPID () = (nextPID := !nextPID + 1; !nextPID)

  (* number of "clock ticks" or loops through the set of threads *)
  val clock = ref 0  
  (* number of times a (meaningful) step has been taken *)
  val work = ref 0   
  fun resetCounts () = (clock := 0; work := 0)
  fun countsToString () =
      "Elapsed clock ticks: " ^ Int.toString(!clock) ^ "\n"
    ^ "         Total work: " ^ Int.toString(!work) ^ "\n"

  fun step (Eval (k, E, (DB.Int i))) = Return (k, E, VInt (i))
    | step (Eval (k, E, (DB.Bool b))) = Return (k, E, VBool (b))

    | step (Eval (k, E, DB.Primop (p, e::es))) = 
      Eval (FPrimopN (p, nil, es)::k, E, e)
    | step (Return (FPrimopN (p, vs, e::es)::k, E, v)) = 
      Eval (FPrimopN (p, vs@[v], es)::k, E, e)
    | step (Return (FPrimopN (p, vs, nil)::k, E, v)) = 
      Return (k, E, case (DB.evalPrimop (p, map expOf (vs@[v])))
                     of SOME (v) => valueOf v
                      | _ => raise Stuck)

    | step (Eval (k, E, DB.If (e, e1, e2))) =
      Eval (FIf (e1, e2)::k, E, e)
    | step (Return (FIf (e1, e2)::k, E, VBool(true))) = Eval (k, E, e1)
    | step (Return (FIf (e1, e2)::k, E, VBool(false))) = Eval (k, E, e2)

    | step (Eval (k, E, DB.Unit)) = Return (k, E, VUnit)

    | step (Eval (k, E, DB.Pair (e1, e2))) = Eval ((FPair1 (e2))::k, E, e1)
    | step (Return ((FPair1 (e2))::k, E, v1)) = Eval ((FPair2 (v1))::k, E, e2)
    | step (Return ((FPair2 (v1))::k, E, v2)) = Return (k, E, VPair (v1, v2))

    | step (Eval (k, E, DB.Fst (e))) = Eval (FFst::k, E, e)
    | step (Return (FFst::k, E, VPair (v1, v2))) = Return (k, E, v1)

    | step (Eval (k, E, DB.Snd (e))) = Eval (FSnd::k, E, e)
    | step (Return (FSnd::k, E, VPair (v1, v2))) = Return (k, E, v2)

    | step (Eval (k, E, e as DB.Fn(_))) = Return (k, E, VClosure (E, e))
    | step (Eval (k, E, DB.Apply (e1, e2))) = Eval (FApply1 (e2)::k, E, e1)
    | step (Return (FApply1 (e2)::k, E, v1)) = Eval (FApply2 (v1)::k, E, e2)

    | step (Return (FApply2 (VClosure (E', DB.Fn(_, (_, e))))::k, E, v2)) =
      Eval (FEnv (E)::k, BOrd (v2)::E', e)

    | step (Eval (k, E, DB.Let(e1, (_, e2)))) = Eval (FLet (e2)::k, E, e1)
    | step (Return (FLet (e2)::k, E, v1)) = 
      Eval (FEnv (E)::k, BOrd (v1)::E, e2)
    | step (Eval (k, E, DB.Var (x))) = 
      let fun lookup x (BOrd (v)::E') = 
              if x = 1 then Return (k, E, v) else lookup (x-1) E'
            | lookup x (BRec (VClosure (E'', e))::E') = 
              if x = 1 then Eval (FEnv (E)::k, E'', e) else lookup (x-1) E'
            | lookup _ _ = raise Stuck
      in 
        lookup x E
      end

    | step (Return (FEnv (E')::k, E, v)) = Return (k, E', v)

    | step (Eval (k, E, e as DB.Rec(_, (_, e1)))) =
      Eval (FEnv (E)::k, BRec (VClosure (E, e))::E, e1)

    | step (Eval (k, E, DB.Leaf)) = Return (k, E, VLeaf)
    | step (Eval (k, E, DB.Node(x, l, r))) = Eval (FNode1(l, r)::k, E, x)
    | step (Return (FNode1(l, r)::k, E, v)) = Eval (FNode2(v, r)::k, E, l)
    | step (Return (FNode2(v, r)::k, E, vl)) = Eval (FNode3(v, vl)::k, E, r)
    | step (Return (FNode3(v, vl)::k, E, vr)) = Return (k, E, VNode(v, vl, vr))
    | step (Eval (k, E, DB.Case(e, e1, (_, _, _, e2)))) =
      Eval (FCase(e1, e2)::k, E, e)
    | step (Return (FCase(e1, e2)::k, E, VLeaf)) = Eval (k, E, e1)
    | step (Return (FCase(e1, e2)::k, E, VNode(x, l, r))) = 
      Eval (FEnv (E)::k, BOrd (r)::BOrd (l)::BOrd (x)::E, e2)

    | step (Return (nil, E, v)) = Done (v)

  (* TODO Future, Promise *)

  fun multiStep (s as Eval(_)) = multiStep (step s)
    | multiStep (s as Return(_)) = multiStep (step s)
    | multiStep (s as Done(_)) = s

  fun stepStream (s as Eval(_)) = 
      S.delay (fn () => S.Cons (s, stepStream (step s)))
    | stepStream (s as Return(_)) = 
      S.delay (fn () => S.Cons (s, stepStream (step s)))
    | stepStream (s as Done(_)) = 
      S.delay (fn () => S.Cons (s, S.delay (fn () => S.Nil)))

  fun evaluate (Eval (k, E, DB.Fn(_, (_, e)))) v = 
      (resetCounts ();
       multiStep (Eval (k, BOrd (v)::E, e)))
    | evaluate _ _ = 
      (print "*** Expected program to be an fn-expression! ***\n";
       raise Stuck)

  fun evaluateStream (Eval (k, E, DB.Fn(_, (_, e)))) v = 
      (resetCounts ();
       stepStream (Eval (k, BOrd (v)::E, e)))
    | evaluateStream _ _ = 
      (print "*** Expected program to be an fn-expression! ***\n";
       raise Stuck)

end;  (* structure EMach *)
