(* Printing expressions *)
(* Supports both named and de Bruijn form *)
signature PRINT =
sig
  type exp

  (* print a type *)
  val typToString : T.typ -> string

  (* print an expression *)
  val expToString : exp -> string
end

signature NAMEDPRINT = PRINT 
    where type exp = MinML.exp

signature DBPRINT = PRINT 
    where type exp = DBMinML.exp

(* new asst4 code: *)
signature MPRINT =
sig
  type lvalue
  type state

  (* print a large value *)
  val lvalueToString : lvalue -> string

  (* print a state *)
  val stateToString : state -> string

end
(* end asst4 code *)

(*--------------------------------------------------*)
structure Print :> NAMEDPRINT =
struct

    structure M = MinML
    type exp = M.exp

    fun po P.Equals = "="
      | po P.LessThan = "<"
      | po P.Plus = "+"
      | po P.Minus = "-"
      | po P.Times = "*"
      | po P.Negate = "~"

    fun typToString T.INT = "int"
      | typToString T.BOOL = "bool"
      | typToString (T.ARROW(d, r)) = 
	"(" ^ typToString d ^ ") -> (" ^ typToString r ^ ")"
(* new asst4 code: *)
      | typToString (T.PAIR(d, r)) = 
	"(" ^ typToString d ^ ") * (" ^ typToString r ^ ")"
      | typToString T.UNIT = "unit"
      | typToString T.EXN = "exn"
(* end asst4 code *)

    and expToString (M.Int i) = Int.toString i
      | expToString (M.Bool true) = "true"
      | expToString (M.Bool false) = "false"
      | expToString (M.If (ec, et, ef)) = 
	"if " ^ expToString ec ^ " then " ^
	expToString et ^ " else " ^ expToString ef ^ " fi"
      | expToString (M.Primop (p, nil)) = "(bad primop)"
      | expToString (M.Primop (p, e::es)) = 
	"(" ^ foldl (fn (a, b) => "(" ^ b ^ ")" ^ po p ^ "(" ^ 
                                  expToString a ^ ")")
	(expToString e) es ^ ")"
      | expToString (M.Fn (t, (x, e))) = 
	"fn " ^ x ^ " : " ^ typToString t ^ " => " ^ expToString e
      | expToString (M.Rec (t, (x, e))) = 
	"rec " ^ x ^ " : " ^ typToString t ^ " => " ^ expToString e
      | expToString (M.Let (e1, (x, e2))) = 
	"let " ^ x ^ " = " ^ expToString e1 ^ " in " ^
	expToString e2 ^ " end"
      | expToString (M.Apply (f, e)) = 
	"(" ^ expToString f ^ " " ^ expToString e ^ ")"
      | expToString (M.Var v) = v
(* new asst4 code: *)
      | expToString (M.Unit) = "()"
      | expToString (M.Pair (e1, e2)) =
	"(" ^ expToString e1 ^ ", " ^ expToString e2 ^ ")"
      | expToString (M.Fst e) = "(fst " ^ expToString e ^ ")"
      | expToString (M.Snd e) = "(snd " ^ expToString e ^ ")"

      | expToString (M.Exception(x,e)) =
	"exception " ^ x ^ " in " ^ expToString e ^ " end "
      | expToString (M.Try(e1,e2,e3)) =
	  "try " ^ expToString e1
	^ " catch " ^ expToString e2
	^ " with " ^ expToString e3
	^ " end "

      | expToString (M.Raise(t, e1)) =
	"(raise[" ^ typToString t ^ "] " ^ expToString e1 ^ ")"
(* end asst4 code *)

end;  (* structure Print *)

structure DBPrint :> DBPRINT =
struct

    structure M = DBMinML
    type exp = M.exp

    fun po P.Equals = "="
      | po P.LessThan = "<"
      | po P.Plus = "+"
      | po P.Minus = "-"
      | po P.Times = "*"
      | po P.Negate = "~"

    fun typToString T.INT = "int"
      | typToString T.BOOL = "bool"
      | typToString (T.ARROW(d, r)) = 
	"(" ^ typToString d ^ ") -> (" ^ typToString r ^ ")"
(* new asst4 code: *)
      | typToString T.UNIT = "unit"
      | typToString (T.PAIR(d, r)) = 
	"(" ^ typToString d ^ ") * (" ^ typToString r ^ ")"
      | typToString T.EXN = "exn"
(* end asst4 code *)

    and expToString (M.Int i) = Int.toString i
      | expToString (M.Bool true) = "true"
      | expToString (M.Bool false) = "false"
      | expToString (M.Fn (t, ((), e))) = 
	"(fn _ : " ^ typToString t ^ " => " ^ expToString e ^ ")"
      | expToString (M.Rec (t, ((), e))) = 
	"(rec _ : " ^ typToString t ^ " => " ^ expToString e ^ ")"
      | expToString (M.If (ec, et, ef)) = 
	"if " ^ expToString ec ^ " then " ^
	expToString et ^ " else " ^ expToString ef ^ " fi"
      | expToString (M.Primop (p, nil)) = "(bad primop)"
      | expToString (M.Primop (p, e::nil)) = "(" ^ po p ^ expToString e ^ ")"
      | expToString (M.Primop (p, e::es)) = 
	"(" ^ foldl (fn (a, b) => b ^ " " ^ po p ^ " " ^ 
                                  expToString a)
	(expToString e) es ^ ")"
      | expToString (M.Let (e1, ((), e2))) = 
	"let - = " ^ expToString e1 ^ " in " ^ expToString e2 ^ " end"
      | expToString (M.Apply (e1, e2)) = 
	"(" ^ expToString e1 ^ " " ^ expToString e2 ^ ")"
      | expToString (M.Var x) = "DB[" ^ (Int.toString x) ^ "]"
(* new asst4 code: *)
      | expToString (M.Unit) = "()"
      | expToString (M.Pair (e1, e2)) =
	"(" ^ expToString e1 ^ ", " ^ expToString e2 ^ ")"
      | expToString (M.Fst e) = "(fst " ^ expToString e ^ ")"
      | expToString (M.Snd e) = "(snd " ^ expToString e ^ ")"

      | expToString (M.Exception(x,e)) =
	"exception " ^ "_" ^ " in " ^ expToString e ^ " end "
      | expToString (M.Try(e1,e2,e3)) =
	  "try " ^ expToString e1
	^ " catch " ^ expToString e2
	^ " with " ^ expToString e3
	^ " end "

      | expToString (M.Raise(t, e1)) =
	"(raise[" ^ typToString t ^ "] " ^ expToString e1 ^ ")"
(* end asst4 code *)

end; (* structure DBPrint *)

(* new asst4 code: *)
functor MPrint (M : MACH) : MPRINT = 
struct
    structure DB = DBPrint
    structure Mem = Memory

    type lvalue = M.lvalue
    type state = M.state

    fun po P.Equals = "="
      | po P.LessThan = "<"
      | po P.Plus = "+"
      | po P.Minus = "-"
      | po P.Times = "*"
      | po P.Negate = "~"

    fun bindingToString (M.BOrd (sv)) = svalueToString sv
      | bindingToString (M.BRec (sv)) = svalueToString sv

    and envToString (M.BOrd (sv)::E) =
        foldl (fn (a, b) => bindingToString a ^ ", " ^ b) (svalueToString sv) E
      | envToString (M.BRec (sv)::E) =
        foldl (fn (a, b) => bindingToString a ^ ", " ^ b) (svalueToString sv) E
      | envToString nil = "-"

    and svalueToString (M.VInt (i)) = Int.toString i
      | svalueToString (M.VBool (true)) = "true"
      | svalueToString (M.VBool (false)) = "false"
      | svalueToString (M.VUnit) = "()"
      | svalueToString (M.VExn (_)) = "EXN"
      | svalueToString (M.VLoc (loc)) = "LOC: " ^ (Mem.locToString loc)

    and lvalueToString (M.VPair (v1, v2)) = "(" ^ svalueToString v1 ^ ", " ^
                                            svalueToString v2 ^ ")"
      | lvalueToString (M.VClosure (E, e)) = "<<" ^ envToString E ^ "; " ^ 
                                             DB.expToString e ^ ">>"

    fun frameToString (M.FPrimopN (p, v::vs, e::es)) =
        po p ^ "(" ^ foldl (fn (a, b) => b ^ ", " ^ svalueToString a)
                           (svalueToString v) vs ^ ", _, " 
        ^ foldl (fn (a, b) => b ^ ", " ^ DB.expToString a)
                (DB.expToString e) es ^ ")"
      | frameToString (M.FPrimopN (p, nil, e::es)) =
        po p ^ "(_, " ^ foldl (fn (a, b) => b ^ ", " ^ DB.expToString a)
                              (DB.expToString e) es ^ ")"
      | frameToString (M.FPrimopN (p, v::vs, nil)) =
        po p ^ "(" ^ foldl (fn (a, b) => b ^ ", " ^ svalueToString a)
                           (svalueToString v) vs ^ ", _)"
      | frameToString (M.FPrimopN (p, nil, nil)) = po p ^ "(_)"
      | frameToString (M.FIf (e1, e2)) = "if(_, " ^ DB.expToString e1 ^ ", " ^
                                         DB.expToString e2 ^ ")"

      | frameToString (M.FPair1 (e2)) = "pair(_, " ^ DB.expToString e2 ^ ")"
      | frameToString (M.FPair2 (v1)) = "pair(" ^ svalueToString v1 ^ ", _)"
      | frameToString (M.FFst) = "fst(_)"
      | frameToString (M.FSnd) = "snd(_)"
      | frameToString (M.FApply1 (e2)) = "apply1(_, " ^ DB.expToString e2 ^ ")"
      | frameToString (M.FApply2 (v1)) = "apply2(" ^ svalueToString v1 ^ ", _)"
      | frameToString (M.FLet (e2)) = "let(" ^ DB.expToString e2 ^ ")"
      | frameToString (M.FRaise) = "raise(_)"
      | frameToString (M.FTry1 (e1, e3)) = "try(" ^ DB.expToString e1 ^
                                           ", _, " ^ DB.expToString e3 ^ ")"
      | frameToString (M.FTry2 (v2, e3)) = "try(_, " ^ svalueToString v2 ^
                                           ", " ^ DB.expToString e3 ^ ")"
      | frameToString (M.FEnv (E)) = "{" ^ envToString E ^ "}"

    fun stackToString (k::ks) =
        foldl (fn (a, b) => frameToString a ^ " |> " ^ b)
              (frameToString k) ks
      | stackToString nil = "-"

    fun stateToString (M.Eval (H, k, E, e)) =
        stackToString k ^ " | " ^ envToString E ^ " > " ^ DB.expToString e
      | stateToString (M.Raise (H, k, E, v)) =
        stackToString k ^ " | " ^ envToString E ^ " << " ^ svalueToString v
      | stateToString (M.Return (H, k, E, v)) = 
        stackToString k ^ " | " ^ envToString E ^ " < " ^ svalueToString v

end;  (* structure MPrint *)

(* end asst4 code *)

