signature MEMORY =
sig
  type loc
  type 'a memory

  (* Memory is raised when we try to allocate but there are no remaining cells
  available. *) 
  exception Memory

  (* Error is raise for all other unexpected conditions (i.e. invalid
  locations) *)
  exception Error of string

  val init : int -> 'a memory
  
  val alloc : 'a memory -> 'a -> loc

  val deref : 'a memory -> loc -> 'a

  (* should be used only be the collector *)
  val assign : 'a memory -> loc -> 'a -> unit
  (* clears a particular cell corresponding to the given location *)
  val clear : 'a memory -> loc -> unit
  (* clears ALL cells in memory *)
  val clearAll : 'a memory -> unit
  val contains : 'a memory -> loc -> bool

  val size : 'a memory -> int
  val free : 'a memory -> int

  val locToString : loc -> string
  val memoryToString : 'a memory -> ('a -> string) -> string

end

structure Memory :> MEMORY =
struct
  open Array

  type loc = int  
  type 'a memory = 'a option array

  exception Memory
  exception Error of string

  fun init size = tabulate (size, fn _ => NONE)
  
  fun alloc H lv = 
      let fun findLoc i =
              case sub (H, i)
                of SOME _ => findLoc (i + 1)
                 | NONE => i
          val loc = findLoc 0
          val _ = update (H, loc, SOME(lv))
      in
        loc
      end
        handle Subscript => raise Memory

  fun deref H loc = 
      (case (sub (H, loc))
           of SOME lv => lv
            | NONE => raise Error "Bad location: free cell")
      handle Subscript => raise Error "Bad Location: out of bounds"

  fun assign H loc lv =
      (case (sub (H, loc))
           of SOME _ => raise Error "Bad location: cannot overwrite"
            | NONE => update (H, loc, SOME(lv)))
      handle Subscript => raise Error "Bad location: out of bounds"

  fun clear H loc =
      update (H, loc, NONE)
      handle Subscript => raise Error "Bad location: out of bounds"

  fun clearAll H =
      modify (fn _ => NONE) H

  fun contains H loc =
      (case (sub (H, loc))
            of SOME _ => true
             | NONE => false)
      handle Subscript => raise Error "Bad location: out of bounds"

  fun size H = length H
  fun free H = foldl (fn (SOME _, n) => n | (NONE, n) => n + 1) 0 H

  fun locToString loc = Int.toString loc
  fun memoryToString H ts = foldri (fn (i, a, s) => 
                                 ("#" ^ Int.toString i ^ ": " ^ (case a 
                                          of SOME(a) => (ts a)
                                           | NONE => "<free>") 
                                  ^ " " ^ s)) "" (H, 0, NONE)
end
