signature MACH =
sig

  type exp = DBMinML.exp

  datatype binding =
      BOrd of svalue             (* ordinary, non-recursive bindings *)
    | BRec of svalue             (* Recusive bindings i.e. =*= *)

  (* svalues or "small values" denote values which would be small enough to
  fit in a register on a typical microprocessor. *)
  and svalue =
      VInt of int
    | VBool of bool
    | VUnit
    | VExn of unit ref
    | VLoc of Memory.loc

  (* lvalues or "large values" denote values that would be allocated on the
  heap. *)
  and lvalue =
      VPair of svalue * svalue
    | VClosure of binding list * exp

  and frame = 
      FPrimopN of P.primop * svalue list * exp list
    | FIf of exp * exp          (* if(_, e1, e2) *)
    | FPair1 of exp             (* pair(_, e2) *)
    | FPair2 of svalue          (* pair(v1,_) *)
    | FFst                      (* first(_) *)
    | FSnd                      (* snd(_) *)
    | FApply1 of exp            (* apply(_, e2) *)
    | FApply2 of svalue         (* apply(v1, _) *)
    | FLet of exp               (* let(_, e2) *)
    | FRaise                    (* raise(_) *)
    | FTry1 of exp * exp        (* try(e1, _, e3) *)
    | FTry2 of svalue * exp     (* try(_, v2, e3) *)
    | FEnv of binding list

  type heap
  type stack = frame list

  datatype state = 
      Eval of heap * stack * binding list * exp
    | Raise of heap * stack * binding list * svalue
    | Return of heap * stack * binding list * svalue

  (* partial function from machine values to expressions *)
  val expOf : svalue -> exp

  (* partial function from expressions to machine values *)
  val valueOf : exp -> svalue

  (* extracts the heap part of a state *)
  val heapOf : state -> heap

  (* extracts the stack part of a state *)
  val stackOf : state -> stack

  (* extracts the current environment from a state *)
  val envOf : state -> binding list

end;  (* signature MACH *)


functor Mach (type heap) : MACH where type heap = heap =
struct
  type exp = DBMinML.exp

  datatype binding =
      BOrd of svalue             (* ordinary, non-recursive bindings *)
    | BRec of svalue             (* Recusive bindings i.e. =*= *)

  (* svalues or "small values" denote values which would be small enough to
  fit in a register on a typical microprocessor. *)
  and svalue =
      VInt of int
    | VBool of bool
    | VUnit
    | VExn of unit ref
    | VLoc of Memory.loc

  (* lvalues or "large values" denote values that would be allocated on the
  heap. *)
  and lvalue =
      VPair of svalue * svalue
    | VClosure of binding list * exp

  and frame = 
      FPrimopN of P.primop * svalue list * exp list
    | FIf of exp * exp          (* if(_, e1, e2) *)
    | FPair1 of exp             (* pair(_, e2) *)
    | FPair2 of svalue          (* pair(v1,_) *)
    | FFst                      (* first(_) *)
    | FSnd                      (* snd(_) *)
    | FApply1 of exp            (* apply(_, e2) *)
    | FApply2 of svalue         (* apply(v1, _) *)
    | FLet of exp               (* let(_, e2) *)
    | FRaise                    (* raise(_) *)
    | FTry1 of exp * exp        (* try(e1, _, e3) *)
    | FTry2 of svalue * exp     (* try(_, v2, e3) *)
    | FEnv of binding list

  type heap = heap
  type stack = frame list

  datatype state = 
      Eval of heap * stack * binding list * exp
    | Raise of heap * stack * binding list * svalue
    | Return of heap * stack * binding list * svalue

  structure DB = DBMinML

  fun expOf (VInt i) = DB.Int (i)
    | expOf (VBool b) = DB.Bool (b)
    | expOf (VUnit) = DB.Unit
    | expOf _ = raise Match

  fun valueOf (DB.Int (i)) = VInt (i)
    | valueOf (DB.Bool (b)) = VBool (b)
    | valueOf (DB.Unit) = VUnit
    | valueOf _ = raise Match

  fun heapOf (Eval (H, _, _, _)) = H
    | heapOf (Raise (H, _, _, _)) = H
    | heapOf (Return (H, _, _, _)) = H

  fun stackOf (Eval (_, k, _, _)) = k
    | stackOf (Raise (_, k, _, _)) = k
    | stackOf (Return (_, k, _, _)) = k

  fun envOf (Eval (_, _, E, _)) = E
    | envOf (Raise (_, _, E, _)) = E
    | envOf (Return (_, _, E, _)) = E

end;


structure AMach : MACH =
struct

  structure DB = DBMinML
  structure Mem = Memory

  type exp = DBMinML.exp

  datatype binding =
      BOrd of svalue             (* ordinary, non-recursive bindings *)
    | BRec of svalue             (* Recusive bindings i.e. =*= *)

  (* svalues or "small values" denote values which would be small enough to
  fit in a register on a typical microprocessor. *)
  and svalue =
      VInt of int
    | VBool of bool
    | VUnit
    | VExn of unit ref
    | VLoc of Memory.loc

  (* lvalues or "large values" denote values that would be allocated on the
  heap. *)
  and lvalue =
      VPair of svalue * svalue
    | VClosure of binding list * exp

  and frame = 
      FPrimopN of P.primop * svalue list * exp list
    | FIf of exp * exp          (* if(_, e1, e2) *)
    | FPair1 of exp             (* pair(_, e2) *)
    | FPair2 of svalue          (* pair(v1,_) *)
    | FFst                      (* first(_) *)
    | FSnd                      (* snd(_) *)
    | FApply1 of exp            (* apply(_, e2) *)
    | FApply2 of svalue         (* apply(v1, _) *)
    | FLet of exp               (* let(_, e2) *)
    | FRaise                    (* raise(_) *)
    | FTry1 of exp * exp        (* try(e1, _, e3) *)
    | FTry2 of svalue * exp     (* try(_, v2, e3) *)
    | FEnv of binding list

  type heap = lvalue Mem.memory
  type stack = frame list

  datatype state = 
      Eval of heap * stack * binding list * exp
    | Raise of heap * stack * binding list * svalue
    | Return of heap * stack * binding list * svalue

  fun expOf (VInt i) = DB.Int (i)
    | expOf (VBool b) = DB.Bool (b)
    | expOf (VUnit) = DB.Unit
    | expOf _ = raise Match

  fun valueOf (DB.Int (i)) = VInt (i)
    | valueOf (DB.Bool (b)) = VBool (b)
    | valueOf (DB.Unit) = VUnit
    | valueOf _ = raise Match

  fun heapOf (Eval (H, _, _, _)) = H
    | heapOf (Raise (H, _, _, _)) = H
    | heapOf (Return (H, _, _, _)) = H

  fun stackOf (Eval (_, k, _, _)) = k
    | stackOf (Raise (_, k, _, _)) = k
    | stackOf (Return (_, k, _, _)) = k

  fun envOf (Eval (_, _, E, _)) = E
    | envOf (Raise (_, _, E, _)) = E
    | envOf (Return (_, _, E, _)) = E

end;  (* structure AMach *)


structure AGMach : MACH =
struct

  structure DB = DBMinML
  structure Mem = Memory

  type exp = DBMinML.exp

  datatype binding =
      BOrd of svalue             (* ordinary, non-recursive bindings *)
    | BRec of svalue             (* Recusive bindings i.e. =*= *)

  (* svalues or "small values" denote values which would be small enough to
  fit in a register on a typical microprocessor. *)
  and svalue =
      VInt of int
    | VBool of bool
    | VUnit
    | VExn of unit ref
    | VLoc of Memory.loc

  (* lvalues or "large values" denote values that would be allocated on the
  heap. *)
  and lvalue =
      VPair of svalue * svalue
    | VClosure of binding list * exp

  and frame = 
      FPrimopN of P.primop * svalue list * exp list
    | FIf of exp * exp          (* if(_, e1, e2) *)
    | FPair1 of exp             (* pair(_, e2) *)
    | FPair2 of svalue          (* pair(v1,_) *)
    | FFst                      (* first(_) *)
    | FSnd                      (* snd(_) *)
    | FApply1 of exp            (* apply(_, e2) *)
    | FApply2 of svalue         (* apply(v1, _) *)
    | FLet of exp               (* let(_, e2) *)
    | FRaise                    (* raise(_) *)
    | FTry1 of exp * exp        (* try(e1, _, e3) *)
    | FTry2 of svalue * exp     (* try(_, v2, e3) *)
    | FEnv of binding list

  (* For the incremental case we must track the from- and to-spaces even while
  the user program is running. *)
  type heap = lvalue Mem.memory * Mem.loc list * lvalue Mem.memory * bool
  type stack = frame list

  datatype state = 
      Eval of heap * stack * binding list * exp
    | Raise of heap * stack * binding list * svalue
    | Return of heap * stack * binding list * svalue

  fun expOf (VInt i) = DB.Int (i)
    | expOf (VBool b) = DB.Bool (b)
    | expOf (VUnit) = DB.Unit
    | expOf _ = raise Match

  fun valueOf (DB.Int (i)) = VInt (i)
    | valueOf (DB.Bool (b)) = VBool (b)
    | valueOf (DB.Unit) = VUnit
    | valueOf _ = raise Match

  fun heapOf (Eval (H, _, _, _)) = H
    | heapOf (Raise (H, _, _, _)) = H
    | heapOf (Return (H, _, _, _)) = H

  fun stackOf (Eval (_, k, _, _)) = k
    | stackOf (Raise (_, k, _, _)) = k
    | stackOf (Return (_, k, _, _)) = k

  fun envOf (Eval (_, _, E, _)) = E
    | envOf (Raise (_, _, E, _)) = E
    | envOf (Return (_, _, E, _)) = E

end;  (* structure AGMach *)
