signature LOOP =
sig
  type 'a action = 'a -> unit
  type state

  (* print the expression *)
  val show      : DBMinML.exp action
  val show_noDB      : MinML.exp action

  (* .. with the stack and the environment *)
  val showState : state action

  (* ... with its type *)
  val showType  : DBMinML.exp action

  (* apply an action to a completely evaluated expression *)
  val eval      : DBMinML.exp action -> DBMinML.exp action

  (* apply an action each to step of the evaluation with given heap size *)
  val step      : int -> state action -> state action

  (* wait after executing an action *)
  val wait      : DBMinML.exp action -> DBMinML.exp action

  (* run an action on user input expressions, without translating to 
     deBruijn form *)
  val loop_noDB      : MinML.exp action -> unit
  (* ... on a file *)
  val loopFile_noDB  : string -> MinML.exp action -> unit

  (* run an action on user input expressions, after translating to 
     deBruijn form *)
  val loop      : DBMinML.exp action -> unit
  (* ... on a file *)
  val loopFile  : string -> DBMinML.exp action -> unit

  (* run an action on user input expressions, after translating to deBruijn
     form and building a initial machine state with given heap size *) 
  val loop_state : int -> state action -> unit

  (* ... on a file *)
  val loopFile_state  : int -> string -> state action -> unit
end

functor Loop (structure M : MACH 
              structure G : GC
  where type lvalue = M.lvalue 
    and type heap = M.heap
    and type state = M.state) =
struct

  structure Eval = Eval (structure M = M structure G = G)
  structure MPrint = MPrint(M)

  type 'a action = 'a -> unit

  fun typing e =
          (case Typing.typeOpt e
             of SOME t => " : " ^ DBPrint.typToString t
              | NONE => " has no type.")

  (* A few actions *)

  fun show e =
      List.app print [DBPrint.expToString e, ";\n"]

  fun show_noDB e =
      List.app print [Print.expToString e, ";\n"]

  fun showState s =
      List.app print [MPrint.stateToString s, ";\n\n",
                      G.heapToString (M.heapOf s), "\n\n"]

  fun showType e =
      List.app print [DBPrint.expToString e, typing e, "\n"]
 
  fun eval heapSize action e = 
      (case (Eval.multiStep (Eval.start heapSize e))
        of M.Return(_, _, _, e') => 
           action (case e'
                    of M.VLoc (i) => raise Match
                     | _ => Eval.expOf e')
         | _ => raise Match) (* impossible or we don't care *)

  fun wait action e =
      (action e;
       print "Press return:";
       TextIO.inputLine TextIO.stdIn;
       ())

  fun step action e = Stream.app action (Eval.stepStream e)

  (* Running the actions on an interactive loop or a file *)

  fun loop action =
         Stream.app action
         ((Translate.translate o Parse.parse o Lexer.lex) 
              (Input.promptkeybd "MinML> "))

  fun loop_noDB action =
         Stream.app action
         (Parse.parse (Lexer.lex (Input.promptkeybd "MinML> ")))

  fun loop_state heapSize action =
      Stream.app action
                 (Stream.map (Eval.start heapSize)
                             ((Translate.translate o Parse.parse o Lexer.lex) 
                                  (Input.promptkeybd "MinML> ")))

  fun loopFile name action =
         Stream.app action
         ((Translate.translate o Parse.parse o Lexer.lex) 
              (Input.readfile name))

  fun loopFile_noDB name action =
         Stream.app action
         (Parse.parse (Lexer.lex (Input.readfile name)))

  fun loopFile_state heapSize name action =
      Stream.app action
                 (Stream.map (Eval.start heapSize)
                             ((Translate.translate o Parse.parse o Lexer.lex) 
                                  (Input.readfile name)))

end
