signature GC =
sig

  (* Raised to indicate that an allocation or dereference has failed (due to
  lack of resources (or breaking of another invariant) and a collection should
  be invoked. *) 
  exception Collect

  (* Types shared between the GC and the abstract machine. *)
  type lvalue
  type heap
  type state

  (* Initialize the GC heap. *)
  val init : int -> heap

  (* Allocate a memory cell from the GC heap and store the given (large)
  value. *)
  val alloc : heap -> lvalue -> Memory.loc

  (* Deference a location from the heap and return the corresponding large
  value. *)
  val deref : heap -> Memory.loc -> lvalue

  (* Invoke the collector. *)
  val collect : state -> state

  (* Convert the heap to a string representation. *)
  val heapToString : heap -> string
end;

(* Holds functioned used by both versions of the collector. *)
functor Common (M : MACH) =
struct
  open M

  exception Unimplemented

  (* Describes the free locations found in small values, environments, large
  values and stacks, respectively. *)
  fun FLSV (sv:svalue) = raise Unimplemented (* TODO *)
  fun FLE (E:binding list) = raise Unimplemented (* TODO *)
  fun FLLV (lv:lvalue) = raise Unimplemented (* TODO *)
  fun FLS (k:stack) = raise Unimplemented (* TODO *)

end

(* A Stop-The-World Collector. *)
structure StopTheWorldGC : GC =
struct
  structure Mem = Memory
  structure C = Common (AMach)
  structure MPrint = MPrint (AMach)

  open AMach

  exception Collect
  exception Unimplemented

  fun init size = Mem.init size

  (* Allocation raises Collect if no memory is available. *)
  fun alloc H lv = Mem.alloc H lv
      handle Mem.Memory => raise Collect

  (* Deference in the stopping collector is straightforward. *)
  fun deref H loc = Mem.deref H loc

  (* Raised by step if no more collection can occur. *)
  exception Done

  (* Make a transition according to the G machine rules.  Performs EXACTLY one
  step.  The state of the collector: on the left the "from-space", in the
  middle the scan set, and on the right, the "to-space". *) 
  fun step (Hf:lvalue Mem.memory, S:Mem.loc list, Ht:lvalue Mem.memory)
      :(lvalue Mem.memory * Mem.loc list * lvalue Mem.memory) =
      raise Unimplemented (* TODO *)

  (* Perform multiple G machine steps until collection is complete. *)
  fun multiStep s = multiStep (step s) handle Done => s

  (* Initialize the collector and send it on its way. *)
  fun collect (s:state):state = raise Unimplemented (* TODO *)

  (* Helpful for debugging? *)
  fun heapToString H = Mem.memoryToString H MPrint.lvalueToString

end;


structure IncrementalGC : GC =
struct
  structure Mem = Memory
  structure C = Common (AGMach)
  structure MPrint = MPrint (AGMach)

  open AGMach

  (* The state of the collector: on the left the "from-space", in the middle
  the frontier, and on the right, the "to-space" and a bit that tells us
  whether we are on or off. *)
  type state = state

  exception Collect
  exception Unimplemented

  fun init size = (Mem.init size, nil, Mem.init size, false)

  fun alloc (Hf, S, Ht, onOff) lv = raise Unimplemented (* TODO *)

  (* We begin with a simple implementation of deref, but you will enhance this
  version later: TODO *)
  fun deref (Hf, _, _, _) loc = Mem.deref Hf loc

  (* Performs EXACTLY one step as defined by the incremental G-machine
  transition rules.  step should only be invoked whene the collector is on. *)
  fun step (Hf, S, Ht, true) = raise Unimplemented (* TODO *)
    | step (_) = raise Match (* Impossible *)

  fun collect (s:state):state = raise Unimplemented (* TODO *)

  (* Helpful for debugging? *)
  fun heapToString (Hf, nil, Ht, true) = 
      "#From: " ^ (Mem.memoryToString Hf MPrint.lvalueToString) ^
      " Scan: - To: " ^ 
      (Mem.memoryToString Ht MPrint.lvalueToString)
    |  heapToString (Hf, loc::S, Ht, true) = 
      "#From: " ^ (Mem.memoryToString Hf MPrint.lvalueToString) ^
      " Scan: " ^ 
      (foldr (fn (l, s) => (Mem.locToString l) ^ ", " ^ s)
             (Mem.locToString loc) S) ^ " To: " ^
      (Mem.memoryToString Ht MPrint.lvalueToString)
    | heapToString (Hf, _, _, false) = 
      Mem.memoryToString Hf MPrint.lvalueToString

end;
