functor Eval (structure M : MACH
              structure G : GC
  where type lvalue = M.lvalue 
    and type heap = M.heap
    and type state = M.state) =
struct
  
  structure S = Stream
  structure DB = DBMinML

  open M
  
  exception Uncaught
  exception Stuck

  fun start size e = Eval (G.init size, nil, nil, e)

  fun step (Eval (H, k, E, (DB.Int i))) = Return (H, k, E, VInt (i))
    | step (Eval (H, k, E, (DB.Bool b))) = Return (H, k, E, VBool (b))

    | step (Eval (H, k, E, DB.Primop (p, e::es))) = 
      Eval (H, FPrimopN (p, nil, es)::k, E, e)
    | step (Return (H, FPrimopN (p, vs, e::es)::k, E, v)) = 
      Eval (H, FPrimopN (p, vs@[v], es)::k, E, e)
    | step (Return (H, FPrimopN (p, vs, nil)::k, E, v)) = 
      Return (H, k, E, case (DB.evalPrimop (p, map M.expOf (vs@[v])))
                     of SOME (v) => valueOf v
                      | _ => raise Stuck)

    | step (Eval (H, k, E, DB.If (e, e1, e2))) =
      Eval (H, FIf (e1, e2)::k, E, e)
    | step (Return (H, FIf (e1, e2)::k, E, VBool (true))) = Eval (H, k, E, e1)
    | step (Return (H, FIf (e1, e2)::k, E, VBool (false))) = Eval (H, k, E, e2)

    | step (Eval (H, k, E, DB.Unit)) = Return (H, k, E, VUnit)

    | step (Eval (H, k, E, DB.Pair (e1, e2))) = 
      Eval (H, (FPair1 (e2))::k, E, e1)
    | step (Return (H, (FPair1 (e2))::k, E, v1)) = 
      Eval (H, (FPair2 (v1))::k, E, e2)
    | step (Return (H, (FPair2 (v1))::k, E, v2)) = raise Stuck (* TODO *)

    | step (Eval (H, k, E, DB.Fst (e))) = Eval (H, (FFst::k), E, e)
    | step (Return (H, (FFst::k), E, VLoc (loc))) = raise Stuck (* TODO*)

    | step (Eval (H, k, E, DB.Snd (e))) = Eval (H, (FSnd::k), E, e)
    | step (Return (H, (FSnd::k), E, VLoc (loc))) = raise Stuck (* TODO *)

    | step (Eval (H, k, E, e as DB.Fn(_))) = raise Stuck (* TODO *)
    | step (Eval (H, k, E, DB.Apply (e1, e2))) = 
      Eval (H, FApply1 (e2)::k, E, e1)
    | step (Return (H, FApply1 (e2)::k, E, v1)) = 
      Eval (H, FApply2 (v1)::k, E, e2)

    | step (Return (H, FApply2 (VLoc (loc))::k, E, v2)) = raise Stuck (* TODO *)
    | step (Eval (H, k, E, DB.Let (e1, (_, e2)))) = 
      Eval (H, FLet (e2)::k, E, e1)
    | step (Return (H, FLet (e2)::k, E, v1)) = 
      Eval (H, FEnv (E)::k, BOrd (v1)::E, e2)
    | step (Eval (H, k, E, DB.Var (x))) = 
      let fun 
          lookup x (BOrd (v)::E') = 
          if x = 1 then Return (H, k, E, v) else lookup (x-1) E'
        | lookup x (BRec (VLoc (loc))::E') = 
          if x = 1 then 
            let val lv = G.deref H loc in
              case lv
               of VClosure (E'', e) => Eval (H, FEnv (E)::k, E'', e)
                | _ => raise Stuck
            end
          else lookup (x-1) E'
        | lookup _ _ = raise Stuck
      in 
        lookup x E
      end

    | step (Return (H, FEnv (E')::k, E, v)) = Return (H, k, E', v)

    | step (Eval (H, k, E, e as DB.Rec (_, (_, e1)))) = raise Stuck (* TODO *)

    | step (Eval (H, k, E, DB.Raise (_, e))) = Eval (H, FRaise::k, E, e)
    | step (Return (H, FRaise::k, E, v)) = Raise (H, k, E, v)

    | step (Eval (H, k, E, DB.Exception (_, e))) = 
      Eval (H, FEnv (E)::k, BOrd (VExn (ref ()))::E, e)

    | step (Eval (H, k, E, DB.Try (e1, e2, e3))) = 
      Eval (H, FTry1 (e1, e3)::k, E, e2)
    | step (Return (H, FTry1 (e1, e3)::k, E, v2)) = 
      Eval (H, FTry2 (v2, e3)::k, E, e1)
    | step (Return (H, FTry2 (_)::k, E, v1)) = Return (H, k, E, v1)

    | step (Raise (H, FTry2 (VExn (i2), e3)::k, E, v as VExn(i))) = 
      if i2 = i then Eval (H, k, E, e3)
      else Raise (H, k, E, v)

    | step (Raise (H, FEnv (E')::k, E, v)) = Raise (H, k, E', v)
    | step (Raise (H, f::k, E, v)) = Raise (H, k, E, v)

    | step _ = raise Stuck


  (* Try to make a step, but fall back on the GC if we need too. *)
  fun step' s = step s handle G.Collect => G.collect s

  fun multiStep (s as Eval(_)) = multiStep (step' s)
    | multiStep (s as Raise(_, f::k, _, _)) = multiStep (step' s)
    | multiStep (s as Raise(_, nil, _, _)) = raise Uncaught
    | multiStep (s as Return(_, f::k, _, _)) = multiStep (step' s)
    | multiStep (s as Return(_, nil, _, _)) = s

  fun stepStream (s as Eval(_)) = 
      S.delay (fn () => S.Cons (s, stepStream (step' s)))
    | stepStream (s as Raise(_, f::k, _, _)) = 
      S.delay (fn () => S.Cons (s, stepStream (step' s)))
    | stepStream (s as Raise(_, nil, _, _)) = 
      S.delay (fn () => S.Cons (s, S.delay (fn () => raise Uncaught)))
    | stepStream (s as Return(_, f::k, _, _)) = 
      S.delay (fn () => S.Cons (s, stepStream (step' s)))
    | stepStream (s as Return(_, nil, _, _)) = 
      S.delay (fn () => S.Cons (s, S.delay (fn () => S.Nil)))

end;
