structure DBUtil =
struct

  (* Environment trimming and De Bruijn re-indexing courtesy of Daniel Wyand
  *)
 
  open DBMinML

  (* exp * int -> bool. true if an expression uses variable DB[n] *)
  fun usesVar (If (e, e1, e2), n) = 
      usesVar (e, n) orelse usesVar (e1, n) orelse usesVar (e2, n)
    | usesVar (Primop (p, el), n) = List.exists (fn x => usesVar (x, n)) el
    | usesVar (Fn (_, (_, e)), n) = usesVar (e, n+1)
    | usesVar (Rec (_, (_, e)), n) = usesVar (e, n+1)
    | usesVar (Let (e1, (_, e2)), n) = 
      usesVar (e1, n) orelse usesVar (e2, n+1)
    | usesVar (Apply (e1, e2), n) = usesVar (e1, n) orelse usesVar (e2, n)
    | usesVar (Var (n'), n) = n = n'
    | usesVar (Pair (e1, e2), n) = usesVar (e1, n) orelse usesVar (e2, n)
    | usesVar (Fst (e), n) = usesVar (e, n)
    | usesVar (Snd (e), n) = usesVar (e, n)
    | usesVar (Exception (_, e), n) = usesVar (e, n+1)
    | usesVar (Try (e1, e2, e3), n) =
      usesVar(e1, n) orelse usesVar (e2, n) orelse usesVar (e3, n)
    | usesVar (Raise (_, e), n) = usesVar (e, n)
    | usesVar (_, _) = false
      
  (* exp * int -> exp. decrements all debruijin indices greater than n, to
  remove variable n. *)
  fun dropVar (If (e, e1, e2), n) = 
      If (dropVar (e, n), dropVar (e1, n), dropVar (e2, n))
    | dropVar (Primop (p, el), n) = 
      Primop (p, map (fn x => dropVar (x, n)) el)
    | dropVar (Fn (t, (x, e)), n) = Fn (t, (x, dropVar (e, n+1)))
    | dropVar (Rec (t, (x, e)), n) = Rec (t, (x, dropVar (e, n+1)))
    | dropVar (Let (e1, (x, e2)), n) = 
      Let (dropVar (e1, n), (x, dropVar (e2, n+1)))
    | dropVar (Apply (e1, e2), n) = 
      Apply (dropVar (e1, n), dropVar (e2, n))
    | dropVar (Var (n'), n) = if n' > n then Var (n'-1) else Var (n')
    | dropVar (Pair (e1, e2), n) = 
      Pair (dropVar (e1, n), dropVar (e2, n))
    | dropVar (Fst (e), n) = Fst (dropVar (e, n))
    | dropVar (Snd (e), n) = Snd (dropVar (e, n))
    | dropVar (Exception (x, e), n) = Exception (x, dropVar (e, n+1))
    | dropVar (Try (e1, e2, e3), n) =
      Try (dropVar (e1, n), dropVar (e2, n), dropVar (e3, n))
    | dropVar (Raise (t, e), n) = Raise (t, dropVar (e, n))
    | dropVar (e, _) = e

  fun trim' (nil, E', e) = (E', e)
    | trim' (v::E, E', e) = 
      let
	  val n = length (v::E)
      in
	  if usesVar (e, n) then trim' (E, v::E', e)
	  else trim' (E, E', dropVar (e, n))
      end 

  (* trim: value list * exp -> value list * exp *)
  (* removes all values from E that arent used in e, and updates *)
  (* DB indices in e to match the new environment. *)
  fun trim (E, e) = trim' (rev E, nil, e)

end;
