(* Type checking MinML *)
(* Operates on de Bruijn form of expressions *)

signature TYPING =
sig

  exception Error of string

  (* typeOf (e) = t for the t such that |- e : t *)
  (* raises Error if no such t exists *)
  val typeOf : DBMinML.exp -> T.typ

  (* typeOpt (e) = SOME(t) for the t such that |- e : t *)
  (* typeOpt (e) = NONE if no such t exists *)
  val typeOpt : DBMinML.exp -> T.typ option

end;  (* signature TYPING *)

structure Typing :> TYPING =
struct

  open T
  open P
  open DBMinML

  exception Error of string

  (* context for typing *)
  infix 1 ++
  fun (ctx ++ x') = x'::ctx
  (* lookup : var -> typ list -> typ *)
  fun lookup x (t::ctx) = if x = 1 then t else lookup (x - 1) ctx
    | lookup x nil = raise Error "Unbound variable"

  (* type checking *)
  (* typeOf' : typ list -> exp -> typ *)
  fun typeOf' _ (Int _) = INT
    | typeOf' _ (Bool _) = BOOL
    | typeOf' ctx (If (e, e1, e2)) =
      let val t = typeOf' ctx e
      in case t 
          of BOOL => let val t1 = typeOf' ctx e1
                         val t2 = typeOf' ctx e2 
                     in
                       if t1 = t2 then t1 else 
                       raise Error "IF branches differ in type"
                     end
           | _ => raise Error "Expected BOOL for IF conditional"
      end
    | typeOf' ctx (Primop (primop, elist)) =
      let val (domain, codomain) = typeOfPrimop primop 
          fun check_type (t::tlist) (e::elist) = 
              if t = typeOf' ctx e then 
                check_type tlist elist 
              else 
                raise Error "Unexpected type of primop argument"
            | check_type nil nil = codomain
            | check_type _ _ = raise Error "Impossible"
      in
        check_type domain elist
      end
    | typeOf' ctx (Fn (t1, (x, e))) =
      let val t2 = typeOf' (ctx++t1) e
      in 
        T.ARROW(t1, t2)
      end
    | typeOf' ctx (Rec (t, (x, e))) =
      let val t' = typeOf' (ctx++t) e
      in 
        if t = t' then t else raise Error "Unexpected type for REC"
      end
    | typeOf' ctx (Let (e1, (x, e2))) =
      let val t1 = typeOf' ctx e1 
      in 
        typeOf' (ctx++t1) e2
      end
    | typeOf' ctx (Apply (e1, e2)) =
      let val t1 = typeOf' ctx e1
      in case t1
          of ARROW (t11, t12)
             => let val t2 = typeOf' ctx e2
                in
                  if t11 = t2 then t12 else 
                  raise Error "Actual type does not match formal type"
                end
           | _ => raise Error "Expected to apply value of ARROW type"
      end 
    | typeOf' ctx (Var x) = lookup x ctx

  fun typeOf e = typeOf' nil e
  fun typeOpt e = SOME (typeOf e) handle Error s => NONE

end;  (* structure Typing *)
