signature LOOP =
sig
  type 'a action = 'a -> unit

  (* print the expression *)
  val show      : DBMinML.exp action
  val show_noDB      : MinML.exp action

  (* .. with the stack and the environment *)
  val showState : EMach.state action

  (* ... with its type *)
  val showType  : DBMinML.exp action

  (* apply an action to a completely evaluated expression *)
  val eval      : DBMinML.exp action -> DBMinML.exp action

  (* apply an action to each step of the evaluation *)
  val step      : EMach.state action -> EMach.state action

  (* wait after executing an action *)
  val wait      : DBMinML.exp action -> DBMinML.exp action

  (* run an action on user input expressions, without translating to 
     deBruijn form *)
  val loop_noDB      : MinML.exp action -> unit
  (* ... on a file *)
  val loopFile_noDB  : string -> MinML.exp action -> unit

  (* run an action on user input expressions, after translating to 
     deBruijn form *)
  val loop      : DBMinML.exp action -> unit
  (* ... on a file *)
  val loopFile  : string -> DBMinML.exp action -> unit

  (* run an action on user input expressions, after translating to 
     deBruijn form and building a initial machine state *)
  val loop_state : EMach.state action -> unit

  (* ... on a file *)
  val loopFile_state  : string -> EMach.state action -> unit
end

structure Loop :> LOOP =
struct

  type 'a action = 'a -> unit

  fun typing e =
          (case Typing.typeOpt e
             of SOME t => " : " ^ DBPrint.typToString t
              | NONE => " has no type.")

  (* A few actions *)

  fun show e =
      List.app print [DBPrint.expToString e, ";\n"]

  fun show_noDB e =
      List.app print [Print.expToString e, ";\n"]

  fun showState s =
      List.app print [StatePrint.stateToString s, ";\n"]

  fun showType e =
      List.app print [DBPrint.expToString e, typing e, "\n"]

  fun eval action e = 
      (case (EMach.multiStep (EMach.start e))
        of EMach.Return(_, _, e') => 
           action (case e'
                    of EMach.VClosure(_) => raise Match (* something better? *)
                     | _ => EMach.expOf e')
         | _ => raise Match) (* impossible or we don't care *)

  fun wait action e =
      (action e;
       print "Press return:";
       TextIO.inputLine TextIO.stdIn;
       ())

  fun step action e = Stream.app action (EMach.stepStream e)

  (* Running the actions on an interactive loop or a file *)

  fun loop action =
         Stream.app action
         ((Translate.translate o Parse.parse o Lexer.lex) 
              (Input.promptkeybd "MinML> "))

  fun loop_noDB action =
         Stream.app action
         (Parse.parse (Lexer.lex (Input.promptkeybd "MinML> ")))

  fun loop_state action =
      Stream.app action
                 (Stream.map EMach.start 
                             ((Translate.translate o Parse.parse o Lexer.lex) 
                                  (Input.promptkeybd "MinML> ")))

  fun loopFile name action =
         Stream.app action
         ((Translate.translate o Parse.parse o Lexer.lex) 
              (Input.readfile name))

  fun loopFile_noDB name action =
         Stream.app action
         (Parse.parse (Lexer.lex (Input.readfile name)))

  fun loopFile_state name action =
      Stream.app action
                 (Stream.map EMach.start 
                             ((Translate.translate o Parse.parse o Lexer.lex) 
                                  (Input.readfile name)))

end
