signature MACH =
sig

  exception Uncaught
  exception Stuck

  type exp

  (* Note that we'd like to define env and frame recursively, but instead we
  use the definition of env whereever it would be found (e.g. FEnv). *)
  (* type env = value list *)

  datatype value =
      VInt of int
    | VBool of bool
    | VUnit
    | VPair of value * value
    | VClosure of unit          (* TODO: give a proper type for closures *)
    | VSuspend of unit          (* TODO: give a proper type for suspends *)
    | VExn of unit              (* TODO: give a proper type for exceptions *)

  and frame = 
      FPrimopN of P.primop * value list * exp list
    | FIf of exp * exp          (* if(_, e1, e2) *)
    | FPair1 of exp             (* pair(_, e2) *)
    | FPair2 of value           (* pair(v1,_) *)
    | FFst                      (* first(_) *)
    | FSnd                      (* snd(_) *)
    | FApply1 of unit           (* TODO: given a proper type for both kinds *)
    | FApply2 of unit              (* of apply frames *)
    | FLet of unit              (* TODO: give a proper type for let frames *)
    | FRaise of unit            (* TODO: some goes for raise frames *) 
    | FTry1 of unit             (* TODO: some goes for try frames *) 
    | FTry2 of unit             (* TODO: some goes for try frames *) 
    | FEnv of value list

  type stack

  datatype state = 
      Eval of stack * value list * exp
    | Raise of stack * value list * value
    | Return of stack * value list * value

  (* partial function from machine values to expressions *)
  val expOf : value -> exp

  (* partial function from expressions to machine values *)
  val valueOf : exp -> value

  (* returns a new state with an empty stack and an empty environment *)
  val start : exp -> state

  (* one-step evaluation, raises Stuck if impossible *)
  val step : state -> state

  (* repeats step as many times as possible *)
  val multiStep : state -> state

  (* a stream of all steps of an evaluation *)
  val stepStream : state -> state Stream.stream

end;  (* signature MACH *)

structure EMach : MACH =
struct

  open T
  open P
  structure DB = DBMinML
  structure S = Stream

  exception Uncaught
  exception Stuck

  type exp = DB.exp

  (* type env = value list *)

  datatype value =
      VInt of int
    | VBool of bool
    | VUnit
    | VPair of value * value
    | VClosure of unit          (* TODO: give a proper type for closures *)
    | VSuspend of unit          (* TODO: give a proper type for suspends *)
    | VExn of unit              (* TODO: give a proper type for exceptions *)

  and frame = 
      FPrimopN of primop * value list * exp list
    | FIf of exp * exp          (* if(_, e1, e2) *)
    | FPair1 of exp             (* pair(_, e2) *)
    | FPair2 of value           (* pair(v1,_) *)
    | FFst                      (* first(_) *)
    | FSnd                      (* snd(_) *)
    | FApply1 of unit           (* TODO: given a proper type for both kinds *)
    | FApply2 of unit              (* of apply frames *)
    | FLet of unit              (* TODO: give a proper type for let frames *)
    | FRaise of unit            (* TODO: some goes for raise frames *) 
    | FTry1 of unit             (* TODO: some goes for try frames *) 
    | FTry2 of unit             (* TODO: some goes for try frames *) 
    | FEnv of value list

  type stack = frame list

  datatype state = 
      Eval of stack * value list * exp
    | Raise of stack * value list * value
    | Return of stack * value list * value

  fun expOf (VInt i) = DB.Int (i)
    | expOf (VBool b) = DB.Bool (b)
    | expOf (VUnit) = DB.Unit
    | expOf (VPair (v1, v2)) = DB.Pair (expOf v1, expOf v2)
    | expOf _ = raise Match

  fun valueOf (DB.Int (i)) = VInt (i)
    | valueOf (DB.Bool (b)) = VBool (b)
    | valueOf (DB.Unit) = VUnit
    | valueOf (DB.Pair (e1, e2)) = VPair (valueOf e1, valueOf e2)
    | valueOf _ = raise Match

  fun start e = Eval (nil, nil, e)

  fun step (Eval (k, E, (DB.Int i))) = Return (k, E, VInt (i))
    | step (Eval (k, E, (DB.Bool b))) = Return (k, E, VBool (b))

    | step (Eval (k, E, DB.Primop (p, e::es))) = 
      Eval (FPrimopN (p, nil, es)::k, E, e)
    | step (Return (FPrimopN (p, vs, e::es)::k, E, v)) = 
      Eval (FPrimopN (p, vs@[v], es)::k, E, e)
    | step (Return (FPrimopN (p, vs, nil)::k, E, v)) = 
      Return (k, E, case (DB.evalPrimop (p, map expOf (vs@[v])))
                     of SOME (v) => valueOf v
                      | _ => raise Stuck)

    | step (Eval (k, E, DB.If (e, e1, e2))) =
      Eval (FIf (e1, e2)::k, E, e)
    | step (Return (FIf (e1, e2)::k, E, VBool(true))) = Eval (k, E, e1)
    | step (Return (FIf (e1, e2)::k, E, VBool(false))) = Eval (k, E, e2)

      (* TODO: extend step for unit, pairs, fst and snd *)
      (* TODO: extend step for let, rec, fn and apply *)
      (* TODO: extend step for exn, exception, and try *)

  fun multiStep (s as Eval(_)) = multiStep (step s)
    | multiStep (s as Raise(f::k, _, _)) = multiStep (step s)
    | multiStep (s as Raise(nil, _, _)) = raise Uncaught
    | multiStep (s as Return(f::k, _, _)) = multiStep (step s)
    | multiStep (s as Return(nil, _, _)) = s

  fun stepStream (s as Eval(_)) = 
      S.delay (fn () => S.Cons (s, stepStream (step s)))
    | stepStream (s as Raise(f::k, _, _)) = 
      S.delay (fn () => S.Cons (s, stepStream (step s)))
    | stepStream (s as Raise(nil, _, _)) = 
      S.delay (fn () => S.Cons (s, S.delay (fn () => raise Uncaught)))
    | stepStream (s as Return(f::k, _, _)) = 
      S.delay (fn () => S.Cons (s, stepStream (step s)))
    | stepStream (s as Return(nil, _, _)) = 
      S.delay (fn () => S.Cons (s, S.delay (fn () => S.Nil)))

end;  (* structure EMach *)
