#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <assert.h>

#define HEAP_MAXLENGTH_INIT 31

struct heap_el 
{
  float val;
  void * el;
};
  
struct heap_ 
{
  int maxlength;
  int length;
  struct heap_el * array;
};

typedef struct heap_ * heap;

heap heap_create()
{
  heap h = malloc(sizeof(struct heap_));
  assert(h);

  h->maxlength = HEAP_MAXLENGTH_INIT;
  h->length = 0;
  h->array = malloc(sizeof(struct heap_el) * h->maxlength);
  assert(h->array);

  return h;
}

void heap_destroy(heap h)
{
  assert(h);
  assert(h->length == 0);
  
  free(h->array);
  free(h);
}

static void heap_switch(heap h, int pos1, int pos2)
{
  float val = h->array[pos2].val;
  void * el = h->array[pos2].el;
  
  h->array[pos2].val = h->array[pos1].val;
  h->array[pos2].el  = h->array[pos1].el;

  h->array[pos1].val = val;
  h->array[pos1].el  = el;  
}

void heap_add(heap h, float val, void * el)
{
  assert(h);

  if (h->length == h->maxlength) {
    h->maxlength = h->maxlength * 2 + 1;
    h->array = realloc(h->array, sizeof(struct heap_el) * h->maxlength);
    assert(h->array);
  }

  {
    int pos = h->length;

    h->length++;

    h->array[pos].val = val; 
    h->array[pos].el = el; 

    while (pos > 0) {
      int newpos = (pos - 1) / 2;
      
      if (h->array[pos].val > h->array[newpos].val) {
	heap_switch(h,pos,newpos);
	pos = newpos;
      } else {
	break;
      }
    }
  }
}

int heap_isempty(heap h)
{
  assert(h);
  return (h->length == 0);
}

void * heap_getmax(heap h)
{
  void * el;

  assert(h);
  assert(h->length > 0);

  el = h->array[0].el;
  
  h->length--;

  if (h->length) {
    int pos = 0;

    h->array[0].val = h->array[h->length].val; 
    h->array[0].el  = h->array[h->length].el; 
      
    while (2 * pos + 1 < h->length) {
      
      int newpos = 2 * pos + 1;
      
      if ((h->array[pos].val < h->array[newpos].val) || 
	  (h->array[pos].val < h->array[newpos+1].val)) {
	if (h->array[newpos].val >= h->array[newpos+1].val) {
	  heap_switch(h,pos,newpos);
	  pos = newpos;
	} else {
	  heap_switch(h,pos,newpos+1);	  
	  pos = newpos+1;
	}
      } else {
	break;
      }
    }
  }
  return el;
}

#ifdef HEAP_TEST

int main(int argc, char * argv[])
{
  int length, num, c;
  int * array;
  heap h;
  if (argc != 3) {
    fprintf(stderr,"Usage: %s <heap-length> <num-tests>\n", argv[0]);
    return 1;
  }
  length = atoi(argv[1]);
  num = atoi(argv[2]);

  if (length <= 0) {
    fprintf(stderr,"Error: length <= 0\n");
    return 1;    
  }

  h = heap_create();
  
  array = malloc(sizeof(int) * length);
  assert(array);

  for (c = 1; c <= num; c++) {
    int i, i1, i2;
    printf("Test %2d:", c);
    for (i = 0; i < length; i++) {
      array[i] = i+1;
    }
    for (i = 0; i < 4 * length; i++) {
      i1 = random() % length;
      i2 = random() % length;
      if (i1 == i2) continue;
      array[i1] = array[i1] + array[i2];
      array[i2] = array[i1] - array[i2];
      array[i1] = array[i1] - array[i2];
    }
    for (i = 0; i < length; i++) {
      printf(" %d", array[i]);
      heap_add(h, -array[i], (void *)array[i]);
    }
    printf("\n------->");
    for (i = 0; i < length; i++) {
      int val = (int)heap_getmax(h);
      printf(" %d", val);
      assert(val == i+1);
    }
    printf("\n");
    assert(heap_isempty(h));
  }

  heap_destroy(h);

  return 0;
}

#endif
