/**CFile***********************************************************************

  FileName    [smMisc.c]

  PackageName [sm]

  Synopsis    [This file contain the main routine for the batch use of NuSMV.]

  Description [This file contain the main routine for the batch use of
  NuSMV. The batch main executes the various model checking steps in a
  predefined order. After the processing of the input file than it
  return to the calling shell.]

  SeeAlso     []

  Author      [Adapted to NuSMV by Marco Roveri]

  Copyright   [
  This file is part of the ``sm'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "smInt.h"
#include "set.h"

static char rcsid[] UTIL_UNUSED = "$Id: smMisc.c,v 1.2 2003/09/23 21:23:26 flerda Exp $";

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [The batch main.]

  Description [The batch main. It first read the input file, than
  flatten the hierarchy. Aftre this preliminar phase it creates the
  boolean variables necessary for the encoding and then start
  compiling the read model into BDD. Now computes the reachable states
  depending if the flag has been set. before starting verifying if the
  properties specified in the model hold or not it computes the
  fairness constraints. You can also activate the reordering and
  also choose to avoid the verification of the properties.]

  SideEffects []

  SeeAlso     []

******************************************************************************/
void smBatchMain()
{
  node_ptr init_expr         = Nil;
  node_ptr invar_expr        = Nil;
  node_ptr trans_expr        = Nil;
  node_ptr procs_expr        = Nil;
  node_ptr justice_expr      = Nil;
  node_ptr compassion_expr   = Nil;
  node_ptr spec_expr         = Nil;
  node_ptr compute_expr      = Nil;
  node_ptr ltlspec_expr      = Nil;
  node_ptr invar_spec_expr   = Nil;

  /* Necessary to have standard behavior in the batch mode */
  util_resetlongjmp();

  /* Parse the input file */
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Parsing file %s ...", get_input_file(options));
    fflush(nusmv_stderr);
  }

  /* 1: read model */
  if (Parser_ReadSMVFromFile(get_input_file(options))) nusmv_exit(1);

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "...done.\n");
    fflush(nusmv_stderr);
  }
  
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Flattening hierarchy ...");
  }
  
  /* 2: flatten hierarchy */
  /* Initializes the module hash */
  init_module_hash();
  /* check_all_implements(parse_tree); */

  /*
    Processing of the parse tree and constructions of all the
    expressions for the state machine(s). Here the expansions are
    performed so that modules and processes are created. The expansion
    of modules is such that the formal parameters (if any) are
    replaced by the actual ones and the machine is replicated.
  */
  Compile_FlattenHierarchy(sym_intern("main"), Nil, &trans_expr, &init_expr,
                           &invar_expr, &spec_expr, &compute_expr,
                           &ltlspec_expr, &invar_spec_expr, &justice_expr,
			   &compassion_expr, &procs_expr, Nil);

  all_symbols = reverse(all_symbols);
  state_variables   = reverse(state_variables);
  input_variables = reverse(input_variables);
  /* We want input variables at the top of the ordering as default */
  all_variables = append_ns(input_variables, state_variables);
  cmp_struct_set_init(cmps, init_expr);
  cmp_struct_set_invar(cmps, invar_expr);
  cmp_struct_set_trans(cmps, trans_expr);
  cmp_struct_set_procs(cmps, procs_expr);
  cmp_struct_set_justice(cmps, reverse(justice_expr));
  cmp_struct_set_compassion(cmps, reverse(compassion_expr));
  cmp_struct_set_spec(cmps, reverse(spec_expr));
  cmp_struct_set_compute(cmps, reverse(compute_expr));
  cmp_struct_set_ltlspec(cmps, reverse(ltlspec_expr));
  cmp_struct_set_invar_spec(cmps, reverse(invar_spec_expr));

  /* We store properties in the DB of properties */
  Prop_FillPropDb(cmp_struct_get_spec(cmps), cmp_struct_get_compute(cmps),
                  cmp_struct_get_ltlspec(cmps), cmp_struct_get_invar_spec(cmps));

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "...done.\n");
  }

  /* If the -lp option is used, list the properties and exit */
  if (opt_list_properties(options) == true) {
    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "Listing the properties in the database...\n");
    }
    Prop_DbPrintAll(nusmv_stdout);

    return;
  }


  /* 3: build variables */
  /* Initializes the build_vars routines */
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Encoding variables ...");
  }

  Compile_EncodeVarsInit();
  /*
    Builds the ADD containing the list of state variables
    (i.e. vars).
    Side effect on symbol_hash.
  */
  Compile_BuildVarsBdd();

  /* Builds the ADD list representing real_state_variables */
  build_real_state_variables();

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "... done.\n");
  }

  /* 4: build model */
  /*
    Checking for the correctness of the state machines which have
    been declared. It should not mention DD manipulation.
  */
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Building Model ...");
  }

  /* We check that the program is a correct NuSMV program and we
     initialize all the DS necessary for the FSM model construction */
  Compile_InitializeBuildModel();

  /* Builds the sexp FSM of the whole read model */
  Prop_MasterSetScalarFsm(Compile_MakeFsmSexpr(all_variables, 
					       cmp_struct_get_justice(cmps),
					       cmp_struct_get_compassion(cmps)));

  /* Write flat scalar model */
  if (get_output_flatten_model_file(options) != NIL(char)) {
    FILE * ofileid = fopen(get_output_flatten_model_file(options), "w");
    if (ofileid == (FILE *)NULL) {
      fprintf(nusmv_stderr, "Unable to open file \"%s\".\n",
	      get_output_flatten_model_file(options));
      nusmv_exit(1);
    }
    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "Writing whole flat model into file \"%s\"...",
	      get_output_flatten_model_file(options));
    }
    Compile_WriteFlatten(ofileid, cmps);
    if (opt_verbose_level_gt(options, 0)) fprintf(nusmv_stderr, "... done.\n");
    fflush(ofileid);
    fclose(ofileid);
  }

  /* Flavio Lerda */
#if 0
  /* Builds the bexp FSM of the whole read model */
  Prop_MasterSetBoolFsm(Compile_MakeFsmBExpr(all_variables, 
					     cmp_struct_get_justice(cmps),
					     cmp_struct_get_compassion(cmps)));

  /* Write flat boolean model */
  if (get_output_boolean_model_file(options) != NIL(char)) {
    FILE * ofileid = fopen(get_output_boolean_model_file(options), "w");
    if (ofileid == (FILE*)NULL) {
      fprintf(nusmv_stderr, "Unable to open file \"%s\".\n",
	      get_output_flatten_model_file(options));
      nusmv_exit(1);
    }
    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "Writing whole boolean model into file \"%s\"...",
	      get_output_boolean_model_file(options));
    }
    Compile_WriteFlattenBool(ofileid, Prop_MasterGetBoolFsm(), cmps);
    if (opt_verbose_level_gt(options, 0)) fprintf(nusmv_stderr, "... done.\n");
    fflush(ofileid);
    fclose(ofileid);
  }
#endif
  /* Flavio Lerda */

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "... done.\n");
  }

  if (opt_bmc_mode(options) == true) {
    /* Flavio Lerda */
    /* Builds the bexp FSM of the whole read model */
    Prop_MasterSetBoolFsm(Compile_MakeFsmBExpr(all_variables, 
	  cmp_struct_get_justice(cmps),
	  cmp_struct_get_compassion(cmps)));

    /* Write flat boolean model */
    if (get_output_boolean_model_file(options) != NIL(char)) {
      FILE * ofileid = fopen(get_output_boolean_model_file(options), "w");
      if (ofileid == (FILE*)NULL) {
	fprintf(nusmv_stderr, "Unable to open file \"%s\".\n",
	    get_output_flatten_model_file(options));
	nusmv_exit(1);
      }
      if (opt_verbose_level_gt(options, 0)) {
	fprintf(nusmv_stderr, "Writing whole boolean model into file \"%s\"...",
	    get_output_boolean_model_file(options));
      }
      Compile_WriteFlattenBool(ofileid, Prop_MasterGetBoolFsm(), cmps);
      if (opt_verbose_level_gt(options, 0)) fprintf(nusmv_stderr, "... done.\n");
      fflush(ofileid);
      fclose(ofileid);
    }
    /* Flavio Lerda */

    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "Entering BMC mode...\n");
    }

    Bmc_Init();

    if (get_prop_no(options) != -1) {
      int prop_no = get_prop_no(options);
      Prop_Ptr prop;

      if (opt_verbose_level_gt(options, 0)) {
	fprintf(nusmv_stderr, "Verifying property %d...\n", prop_no);
      }
      
      if ( (prop_no < 0) || (prop_no >= Prop_Db_GetSize()) ) {
	fprintf(nusmv_stderr, 
		"Error: \"%d\" is not a valid property index\n", prop_no);
	nusmv_exit(1);
      }

      prop = Prop_Db_GetNum(prop_no);
      
      if (Prop_GetType(prop) == Prop_Ltl) {
	int rel_loop = 
	  Bmc_Utils_ConvertLoopFromString(get_bmc_pb_loop(options), NULL);
	Bmc_GenSolveLtl(prop, 
			get_bmc_pb_length(options), 
			rel_loop, 
			TRUE, /*increases length*/
			SOLVE,  
			NULL);
      } 
      else if (Prop_GetType(prop) == Prop_Invar) {
	Bmc_GenSolveInvar(prop, SOLVE, NULL);
      }
      else {
	fprintf(nusmv_stderr, 
		"Error: only LTL and INVAR properties can be checked in BMC mode\n");
	nusmv_exit(1);
      }
    }
    else {
      if (opt_verbose_level_gt(options, 0)) {
	fprintf(nusmv_stderr, "Verifying the LTL properties...\n");
      }

      {
	lsList props = Prop_Db_GetPropsOfType(Prop_Ltl);
	lsGen  iterator; 
	Prop_Ptr prop;
	int rel_loop;

	nusmv_assert(props != LS_NIL);

	lsForEachItem(props, iterator, prop) {
	  rel_loop = 
	    Bmc_Utils_ConvertLoopFromString(get_bmc_pb_loop(options), NULL);
	  Bmc_GenSolveLtl(prop, 
			  get_bmc_pb_length(options), 
			  rel_loop, 
			  TRUE, /*increases length*/
			  SOLVE, NULL);
	}

	lsDestroy(props, NULL); /* the list is no longer needed */
      }

      if (opt_verbose_level_gt(options, 0)) {
	fprintf(nusmv_stderr, "Verifying the INVAR properties...\n");
      }
      
      {
	lsList props = Prop_Db_GetPropsOfType(Prop_Invar);
	lsGen  iterator; 
	Prop_Ptr prop;
	
	nusmv_assert(props != LS_NIL);
	
	lsForEachItem(props, iterator, prop) {
	  Bmc_GenSolveInvar(prop, SOLVE, NULL);
	}

	lsDestroy(props, NULL); /* the list is no longer needed */
      }
    
    }

    return;
  }

  /* Builds the BDD FSM of the whole read model */
  if (opt_cone_of_influence(options) == false) {
    add_ptr one = add_one(dd_manager);

    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "Building BDD Model ...");
    }

    Prop_MasterSetBddFsm(Compile_MakeFsmBdd(all_variables, 
					    cmp_struct_get_justice(cmps), 
					    cmp_struct_get_compassion(cmps), 
					    one));

    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "... done.\n");
    }

    add_free(dd_manager, one);
  }

  if (opt_cone_of_influence(options) == false) {
    /* Reachability */
    /* Builds the OBDD representing Reachable States Set */
    if (opt_forward_search(options)) {
      if (opt_verbose_level_gt(options, 0))
        fprintf(nusmv_stderr,"Computing reachable states ...");

      compute_reachable_states(Prop_MasterGetBddFsm());

      if (opt_verbose_level_gt(options, 0))
        fprintf(nusmv_stderr,"...done\n");
    }
  }
  
  if (opt_check_trans(options) == true) {
    if (opt_cone_of_influence(options) == false) {
      check_transition_relation(Prop_MasterGetBddFsm());
    }
    else {
      fprintf(nusmv_stderr, "WARNING: Check for totality of the transition relation cannot currently\n");
      fprintf(nusmv_stderr, "performed in batch mode if the cone of influence reduction has been enabled.\n");
    }
  }

  if (get_prop_no(options) != -1) {
    int prop_no = get_prop_no(options);

    if (opt_verbose_level_gt(options, 0)) {
      fprintf(nusmv_stderr, "Verifying property %d...\n", prop_no);
    }
      
    if ( (prop_no < 0) || (prop_no >= Prop_Db_GetSize()) ) {
      fprintf(nusmv_stderr, 
	      "Error: \"%d\" is not a valid property index\n", prop_no);
      nusmv_exit(1);
    }

    Prop_Db_VerifyPropIndex(prop_no);
  }
  else {
    /* Evaluates the Specifications */
    if (!opt_ignore_spec(options)) {
      Prop_DbVerifyAllTypeCtl();
    }

    if (!opt_ignore_compute(options)) {
      Prop_DbVerifyAllTypeCompute();
    }

    /* Evaluates the LTL specifications */
    if (!opt_ignore_ltlspec(options)) {
      Prop_DbVerifyAllTypeLtl();
    }

    /* Evaluates CHECKINVARIANTS */
    if (!opt_ignore_invar(options)) {
      Prop_DbVerifyAllTypeInvar();
    }
  }

  /* Reporting of statistical information. */
  //if (opt_verbose_level_gt(options, 0)) {  
    print_usage(nusmv_stdout);
  //}

  /* Computing and Reporting of the Effect of Reordering */
  if (opt_reorder(options)) {
    fprintf(nusmv_stdout, "\n========= starting reordering ============\n");
    dd_reorder(dd_manager, get_reorder_method(options), DEFAULT_MINSIZE);
    /*    Compile_WriteOrder(get_output_order_file(options), 0); */
    fprintf(nusmv_stdout, "\n========= after reordering ============\n");
    if (opt_verbose_level_gt(options, 0)) {  
      print_usage(nusmv_stdout);
    }
  }
  
  /* Ordering file dump: */
  Compile_WriteOrder(get_output_order_file(options), 0);

  if (opt_print_reachable(options) == true) {
    /* Reporting of Reachable States */
    if (opt_cone_of_influence(options) == false) {
      print_reachable_states(Prop_MasterGetBddFsm(),0);
    }
    else {
      fprintf(nusmv_stderr, "WARNING: Statistics of reachable states is not currently available\n");
      fprintf(nusmv_stderr, "in batch mode if cone of influence reduction has been enabled.\n");
    }
  }
}

/**Function********************************************************************

  Synopsis           [Prints usage statistic.]

  Description        [Prints on <code>nusmv_stdout</code> usage
  statistics, i.e. the amount of memory used, the amount of time
  spent, the number of BDD nodes allocated and the size of the model
  in BDD.]

  SideEffects        []

  SeeAlso            [compilePrintBddModelStatistic]
******************************************************************************/
void print_usage(FILE * file)
{
  fprintf(nusmv_stdout, "######################################################################\n");
  util_print_cpu_stats(file);
  fprintf(file, "######################################################################\n");
  fprintf(file, "BDD statistics\n");
  fprintf(file, "--------------------\n");
  fprintf(file, "BDD nodes allocated: %d\n", get_dd_nodes_allocated(dd_manager));
  fprintf(file, "--------------------\n");
  if (opt_cone_of_influence(options) == false) {
    if (Prop_MasterGetBddFsm() != NULL) {
      compilePrintBddModelStatistic(file, Prop_MasterGetBddFsm());
    }
  }
  else {
    fprintf(nusmv_stderr, "WARNING: Model Statistics is not currently available\n");
    fprintf(nusmv_stderr, "if cone of influence reduction has been enabled.\n");
  }
}

