/**CFile***********************************************************************

  FileName    [simulateCmd.c]

  PackageName [simulate]

  Synopsis    [Model Checker Simulator Commands]

  Description [This file contains commands to be used for the simulation feature.]

  SeeAlso     [simulate.c]

  Author      [Andrea Morichetti]

  Copyright   [
  This file is part of the ``simulate'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "simulateInt.h"
#include "error.h" /* for CATCH */

static char rcsid[] UTIL_UNUSED = "$Id: simulateCmd.c,v 1.1.1.1 2003/02/06 19:01:18 flerda Exp $";

/* Prototypes of command functions */
int CommandSimulate   ARGS((int argc, char **argv));
int CommandShowTraces ARGS((int argc, char **argv));
int CommandPickState  ARGS((int argc, char **argv));

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static int UsageSimulate   ARGS((void));
static int UsageShowTraces ARGS((void));
static int UsagePickState  ARGS((void));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis           [Initializes the simulate package.]

  Description        [Initializes the simulate package.]

  SideEffects        []

******************************************************************************/
void Simulate_Init(void){
  Cmd_CommandAdd("simulate", CommandSimulate, 0);
  Cmd_CommandAdd("show_traces", CommandShowTraces, 0);
  Cmd_CommandAdd("pick_state", CommandPickState, 0);
}

/**Function********************************************************************

  Synopsis           [Quits the simulate package]

  Description        [Quits the simulate package]

  SideEffects        []

******************************************************************************/
void Simulate_End(void)
{ }

/**Function********************************************************************

  Synopsis           [Picks a state from the set of initial states]

  CommandName        [pick_state]

  CommandSynopsis    [Picks a state from the set of initial states]

  CommandArguments   [\[-h\] \[-v\] \[-r | -i \[-a\]\] \[-c "constraints"\]]

  CommandDescription [

  Chooses an element from the set of initial states, and makes it the
  <tt>current state</tt> (replacing the old one). The chosen state is
  stored as the first state of a new trace ready to be lengthened by
  <tt>steps</tt> states by the <tt>simulate</tt> command. The state can be
  chosen according to different policies which can be specified via command
  line options. By default the state is chosen in a deterministic way.
  <p>
  Command Options:<p>
  <dl>
    <dt> <tt>-v</tt>
       <dd> Verbosely prints out chosen state (all state variables, otherwise
       it prints out only the label <tt>t.1</tt> of the state chosen, where
       <tt>t</tt> is the number of the new trace, that is the number of
       traces so far generated plus one).
    <dt> <tt>-r</tt>
       <dd> Randomly picks a state from the set of initial states.
    <dt> <tt>-i</tt>
       <dd> Enables the user to interactively pick up an initial state. The
       user is requested to choose a state from a list of possible items
       (every item in the list doesn't show state variables unchanged with
       respect to a previous item). If the number of possible states is too
       high, then the user has to specify some further constraints as
       "simple expression".
    <dt> <tt>-a</tt>
       <dd> Displays all state variables (changed and unchanged with respect
       to a previous item) in an interactive picking. This option
       works only if the <tt>-i</tt> options has been specified.
    <dt> <tt>-c "constraints"</tt>
       <dd> Uses <tt>constraints</tt> to restrict the set of initial states
       in which the state has to be picked. <tt>constraints</tt> must be
       enclosed between double quotes <tt>" "</tt>.
  </dl> ]

  SideEffects        [The state chosen is stored in the traces_hash table as
  the first state of a new trace]

  SeeAlso            [goto_state simulate]

******************************************************************************/
int CommandPickState(int argc, char ** argv)
{
  int res;
  int c = 0;
  boolean verbose = false;
  int display_all = 1;
  char *strConstr = NIL(char);
  Simulation_Mode mode = Deterministic;
  short int usedMode = 0;


  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "hriavc:")) != EOF) {
    switch(c){
    case 'h': return(UsagePickState());
    case 'r':
      if (++usedMode > 1) return(UsagePickState());
      mode = Random;
      break;
    case 'i':
      if (++usedMode > 1) return(UsagePickState());
      mode = Interactive;
      break;
    case 'a':
      display_all = 0;
      break;
    case 'v':
      verbose = true;
      break;
    case 'c':
      strConstr = util_strsav(util_optarg);
      break;
    default: return(UsagePickState());
    }
  }

  if ((mode != Interactive) && (display_all == 0)) return(UsagePickState());
  if (argc != util_optind) return(UsagePickState());

  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr, "A model must be read before. Use the \"read_model\" command.\n");
    return(1);
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr, "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return(1);
  }
  if (cmp_struct_get_encode_variables(cmps) == 0) {
    fprintf(nusmv_stderr, "The variables must be built before. Use the \"encode_variables\" command.\n");
    return(1);
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr, "A model must be built before. Use the \"build_model\" command.\n");
    return(1);
  }

  res = Simulate_CmdPickOneState(Prop_MasterGetBddFsm(), mode,
                                 display_all, strConstr);

  if (verbose == true) Cmd_CommandExecute("print_current_state -v");
  return(res);
}

static int UsagePickState () {
  fprintf(nusmv_stderr, "usage: pick_state [-h] [-v] [-r | -i [-a]] [-c \"constr\"]\n");
  fprintf(nusmv_stderr, "  -h \t\tPrints the command usage.\n");
  fprintf(nusmv_stderr, "  -v \t\tVerbosely prints picked state.\n");
  fprintf(nusmv_stderr, "  -r \t\tRandomly picks a state from the set of the initial states\n");
  fprintf(nusmv_stderr, "     \t\t(otherwise choice is deterministic).\n");
  fprintf(nusmv_stderr, "  -i \t\tLets the user interactively pick a state from\n");
  fprintf(nusmv_stderr, "     \t\tthe set of initial ones.\n");
  fprintf(nusmv_stderr, "  -a \t\tDisplays all the state variables (changed and\n");
  fprintf(nusmv_stderr, "   \t\tunchanged) in an interactive session.\n");
  fprintf(nusmv_stderr, "   \t\tIt works only together with -i option.\n");
  fprintf(nusmv_stderr, "  -c \"constr\"   Sets constraints for the initial set of states.\n");
  fprintf(nusmv_stderr, "     \t\tNote: must be enclosed between double quotes \" \".\n");
  return(1);
}

/**Function********************************************************************

  Synopsis           [Shows the traces generated in a NuSMV session]

  SeeAlso            [pick_state goto_state simulate]

  CommandName        [show_traces]

  CommandSynopsis    [Shows the traces generated in a NuSMV session]

  CommandArguments   [\[ \[-h\] \[-v\] \[-m | -o output-file\]
  -t | -a | trace_number \]]

  CommandDescription [
  Shows the traces currently stored in system memory, if any. By default
  it shows the last generated trace, if any.
  <p>
  Command Options:<p>
  <dl>
    <dt> <tt>-v</tt>
       <dd> Verbosely prints traces content (all state variables, otherwise
       it prints out only those variables that have changed their value from
       previous state).
    <dt> <tt>-t</tt>
       <dd> Prints only the total number of currently stored traces.
    <dt> <tt>-a</tt>
       <dd> Prints all the currently stored traces.
    <dt> <tt>-m</tt>
       <dd> Pipes the output through the program specified
       by the <tt>PAGER</tt> shell variable if defined, else through the
       <tt>UNIX</tt> command "more".
    <dt> <tt>-o output-file</tt>
       <dd> Writes the output generated by the command to <tt>output-file</tt>
    <dt> <tt>trace_number</tt>
       <dd> The (ordinal) identifier number of the trace to be printed.
    </dl> ]

  SideEffects        []

******************************************************************************/
int CommandShowTraces(int argc, char **argv)
{
  int res = 0;
  int c = 0;
  boolean all = false;
  boolean number = false;
  short int useMore = 0;
  char * dbgFileName = NIL(char);
  char *err_occ[2];
  char *steps_no_str;
  int verbose = 1;
  int traceno = get_trace_number()-1;
  int trace = traceno;
#if HAVE_GETENV
  char * pager;
#endif
FILE * old_nusmv_stdout = NIL(FILE);

  util_getopt_reset();
  while((c = util_getopt(argc,argv,"hvatmo:")) != EOF){
    switch(c){
    case 'h': return(UsageShowTraces());
    case 'v':
      verbose = 0;
      break;
    case 'a':
      all = true;
      break;
    case 't':
      number = true;
      break;
    case 'o':
      if (useMore == 1) return(UsageShowTraces());
      dbgFileName = util_strsav(util_optarg);
      break;
    case 'm':
      if (dbgFileName != NIL(char)) return(UsageShowTraces());
      useMore = 1;
      break;
    default: return(UsageShowTraces());
    }
  }
  if (traceno == 0) {
    fprintf(nusmv_stderr, "There is no trace currently available.\n");
    return(0);
  }
  if ((util_optind == 0) && (argc > 2)) return(UsageShowTraces());

  /* Parsing of the trace number to be printed */
  if (all == false) {
    if (argc != util_optind) {
      err_occ[0] = "";
      steps_no_str = util_strsav(argv[util_optind]);
      trace = strtol(steps_no_str, err_occ, 10);
      if  ((strcmp(err_occ[0], "") != 0)) {
	fprintf(nusmv_stderr,
		"Error: \"%s\" is not a valid value (must be a positive integer).\n", err_occ[0]);
	return(1);
      }
      if ((trace > traceno) || (trace <= 0)) {
	fprintf(nusmv_stderr,
		"Error: \"%d\" is not a valid trace number. Correct range is 1..%d.\n",	trace, traceno);
	return(1);
      }
    }
  }
  else { if (argc != util_optind) return(UsageShowTraces());}

  if (useMore) {
    old_nusmv_stdout = nusmv_stdout;
#if HAVE_GETENV
    pager = getenv("PAGER");
    if (pager == NULL) {
      nusmv_stdout = popen("more", "w");
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
    else {
      nusmv_stdout = popen(pager, "w");
      if (nusmv_stdout == NULL) {
        fprintf(nusmv_stderr, "Unable to open pipe with \"%s\".\n", pager);
        nusmv_stdout = old_nusmv_stdout;
        return(1);
      }
    }
#else
    nusmv_stdout = popen("more", "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open pipe with \"more\".\n");
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
#endif
  }
  if (dbgFileName != NIL(char)) {
    fprintf(nusmv_stdout, "Output to file: %s\n", dbgFileName);
    old_nusmv_stdout = nusmv_stdout;
    nusmv_stdout = fopen(dbgFileName, "w");
    if (nusmv_stdout == NULL) {
      fprintf(nusmv_stderr, "Unable to open file \"%s\".\n", dbgFileName);
      nusmv_stdout = old_nusmv_stdout;
      return(1);
    }
  }

  if (number == true) {
    fprintf(nusmv_stdout, (traceno == 1) ? 
	    "There is %d trace currently available.\n" :
	    "There are %d traces currently available.\n", traceno);
  }
  else {
    set_indent_size(2);
    res = Simulate_CmdShowTraces(trace, verbose, all, dbgFileName);
    reset_indent_size();
    if (useMore) {
      pclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
    if (dbgFileName != NIL(char)) {
      fflush(nusmv_stdout);
      fclose(nusmv_stdout);
      nusmv_stdout = old_nusmv_stdout;
    }
  }

  return res;
}

static int UsageShowTraces () {
  fprintf(nusmv_stderr, "usage: show_traces [-h] [-v] [-t] [-m | -o file] [-a | trace_number]\n");
  fprintf(nusmv_stderr, "  -h \t\tPrints the command usage.\n");
  fprintf(nusmv_stderr, "  -v \t\tVerbosely prints traces content (unchanged vars also).\n");
  fprintf(nusmv_stderr, "  -a \t\tPrints all the currently stored traces.\n");
  fprintf(nusmv_stderr, "  -t \t\tPrints only the total number of currently stored traces.\n");
  fprintf(nusmv_stderr, "  -m \t\tPipes output through the program specified by the \"PAGER\"\n");
  fprintf(nusmv_stderr, "     \t\tenvironment variable if defined, else through the UNIX command \"more\".\n");
  fprintf(nusmv_stderr, "  -o file\tWrites the generated output to \"file\".\n");
  fprintf(nusmv_stderr, "  trace_number\tThe number of the trace to be printed.\n");
  return(1);
}

/**Function********************************************************************

  Synopsis           [Performs a simulation from the current selected state]

  SideEffects        [Generated referenced states traces are stored to be
  analyzed by the user in a second time]

  SeeAlso            [pick_state goto_state]

  CommandName        [simulate]

  CommandSynopsis    [Performs a simulation from the current selected state]

  CommandArguments   [\[-h\] \[-p | -v\] \[-r | -i \[-a\]\] 
  \[-c "constraints"\] steps
  ]

  CommandDescription [
  Generates a sequence of at most <tt>steps</tt> states (representing a
  possible execution of the model), starting from the <em>current state</em>. 
  The current state must be set via the <em>pick_state</em> or 
  <em>goto_state</em> commands.<p> 
  It is possible to run the simulation in three ways (according to different 
  command line policies):
  deterministic (the default mode), random and interactive.<p>
  The resulting sequence is stored in a trace indexed with an integer number
  taking into account the total number of traces stored in the system. There is
  a different behavior in the way traces are built, according to how 
  <em>current state</em> is set: <em>current state</em> is always put at 
  the beginning of a new trace (so it will contain at most <it>steps + 1</it> 
  states) except when it is the last state of an existent old trace. 
  In this case the old trace is lengthened by at most <it>steps</it> states.
  <p>
  Command Options:<p>
  <dl>
    <dt> <tt>-p</tt>
       <dd> Prints current generated trace (only those variables whose value 
       changed from the previous state).
    <dt> <tt>-v</tt>
       <dd> Verbosely prints current generated trace (changed and unchanged 
       state variables).
    <dt> <tt>-r</tt>
       <dd> Picks a state from a set of possible future states in a random way.
    <dt> <tt>-i</tt>
       <dd> Enables the user to interactively choose every state of the trace,
       step by step. If the number of possible states is too high, then 
       the user has to specify some constraints as simple expression. 
       These constraints are used only for a single simulation step and 
       are <em>forgotten</em> in the following ones. They are to be intended 
       in an opposite way with respect to those constraints eventually entered 
       with the pick_state command, or during an interactive simulation 
       session (when the number of future states to be displayed is too high),
       that are <em>local</em> only to a single step of the simulation and 
       are <em>forgotten</em> in the next one.
    <dt> <tt>-a</tt>
       <dd> Displays all the state variables (changed and unchanged) during 
       every step of an interactive session. This option works only if the
       <tt>-i</tt> option has been specified.
    <dt> <tt>-c "constraints"</tt>
       <dd> Performs a simulation in which computation is restricted to states
       satisfying those <tt>constraints</tt>. The desired sequence of states 
       could not exist if such constraints were too strong or it may happen 
       that at some point of the simulation a future state satisfying those
       constraints doesn't exist: in that case a trace with a number of 
       states less than <tt>steps</tt> trace is obtained.
       Note: <tt>constraints</tt> must be enclosed between double quotes 
       <tt>" "</tt>.
    <dt> <tt>steps</tt>
       <dd> Maximum length of the path according to the constraints. 
       The length of a trace could contain less than <tt>steps</tt> states: 
       this is the case in which simulation stops in an intermediate 
       step because it may not exist any future state satisfying those 
       constraints.
    </dl> ]

******************************************************************************/
int CommandSimulate(int argc, char **argv)
{
  bdd_ptr bdd_constraints = (bdd_ptr)NULL;
  boolean isconstraint = false;
  boolean printrace = false;
  int display_all = 1;
  int c = 0;
  int verbose = 1;
  int steps = 0;
  char * strConstr[2]; /* the string array of constraints to parsificate */
  Simulation_Mode mode = Deterministic;

  strConstr[0] = NIL(char);
  util_getopt_reset();
  while((c = util_getopt(argc,argv,"c:hpvria")) != EOF){
    switch(c){
    case 'h': return(UsageSimulate());
    case 'p':
      if (printrace == true) return(UsageSimulate());
      printrace = true;
      verbose = 1;
      break;
    case 'v':
      if (printrace == true) return(UsageSimulate());
      printrace = true;
      verbose = 0;
      break;
    case 'r':
      if (mode == Interactive) return(UsageSimulate());
      mode = Random;
      break;
    case 'i':
      if (mode == Random) return(UsageSimulate());
      mode = Interactive;
      break;
    case 'a':
      display_all = 0;
      break;
    case 'c':
      isconstraint = true;
      strConstr[1] = util_strsav(util_optarg);
      break;
    default:
      return(UsageSimulate());
    }
  }

  if ((mode != Interactive) && (display_all == 0)) return(UsageSimulate());

  /* controls of commands hierarchy */
  if (cmp_struct_get_read_model(cmps) == 0) {
    fprintf(nusmv_stderr,
	    "A model must be read before. Use the \"read_model\" command.\n");
    return(1);
  }
  if (cmp_struct_get_flatten_hrc(cmps) == 0) {
    fprintf(nusmv_stderr,
	    "The hierarchy must be flattened before. Use the \"flatten_hierarchy\" command.\n");
    return(1);
  }
  if (cmp_struct_get_encode_variables(cmps) == 0) {
    fprintf(nusmv_stderr,
	    "The variables must be built before. Use the \"encode_variables\" command.\n");
    return(1);
  }
  if (cmp_struct_get_build_model(cmps) == 0) {
    fprintf(nusmv_stderr,
	    "A model must be built before. Use the \"build_model\" command.\n");
    return(1);
  }

  if (Img_CheckNoCone() == 0) {
    fprintf(nusmv_stderr,
	    "The current partition method %s has not yet be computed.\n",
	    method2str(get_partition_method(options)));
    fprintf(nusmv_stderr, "Use \t \"build_model -f -m %s\"\nto build the partitioned transitioned relation.\n", method2str(get_partition_method(options)));
    return(1);
  }

  if ((argc - util_optind) != 1) return(UsageSimulate()); 

  if (!(current_state_exist())) {
    fprintf(nusmv_stderr,
	    "No current state set. Use the \"pick_state\" command.\n");
    return(1);
  }

  if (isconstraint == true) {
    node_ptr parsed_command = Nil;
    
    if (Parser_ReadCmdFromString(2, strConstr, "CONSTRAINT ", ";\n",
				 &parsed_command) == 0) {
      CATCH {
	if ((parsed_command != Nil) &&
	    (node_get_type(parsed_command) == CONSTRAINT)) {
	  node_ptr constraints = car(parsed_command);
	  add_ptr add_constraints = eval(constraints, Nil);
	  
	  bdd_constraints = add_to_bdd(dd_manager, add_constraints);
	  add_free(dd_manager, add_constraints);
	}
      } /* end CATCH */
      FAIL {
	fprintf(nusmv_stderr,
		"Parsing error: constraints have to be \"simple_expr\".\n");
	return(1);
      }
    }
    else {
      fprintf(nusmv_stderr,
	      "Parsing error: constraints have to be \"simple_expr\".\n");
      return(1);
    }
  }
  else {
    bdd_constraints = bdd_one(dd_manager);
  }

  {
    char *err_occ[1];
    char *steps_no_str;

    err_occ[0] = "";

    steps_no_str = util_strsav(argv[util_optind]);
 
    steps = strtol(steps_no_str, err_occ, 10);
    if  ((strcmp(err_occ[0], "") != 0)) {
      fprintf(nusmv_stderr, \
	      "Error: \"%s\" is not a valid value (must be a positive integer).\n", err_occ[0]);
      bdd_free(dd_manager, bdd_constraints);
      return(1);
    }
  }
  {
    node_ptr current_trace = Nil;
    Fsm_BddPtr fsm;

    fsm = Prop_MasterGetBddFsm();

    fprintf(nusmv_stdout, "********  Simulation Starting From State  ");
    print_node(nusmv_stdout, current_state_label_get());
    fprintf(nusmv_stdout, "  ********\n");
    /* Important: the simulation ALWAYS starts from the current selected state */
    current_trace = Simulate_MultipleSteps(fsm, bdd_constraints, mode, steps, display_all);
    if (current_trace == Nil) {
      bdd_free(dd_manager, bdd_constraints);
      return(1);
    }
    /* prints the trace and inserts it in the traces hash, then frees it*/
    store_and_print_trace(current_trace, printrace, verbose);
    walk_dd(dd_manager, bdd_free, current_trace);
    bdd_free(dd_manager, bdd_constraints);
  }
  return(0);
}

static int UsageSimulate(){
  fprintf(nusmv_stderr, "usage: simulate [-h] [-p | -v] [-r | -i [-a]] [-c \"constr\"] steps\n");
  fprintf(nusmv_stderr, "  -h \t\tPrints the command usage.\n");
  fprintf(nusmv_stderr, "  -p \t\tPrints current generated trace (only changed variables).\n");
  fprintf(nusmv_stderr, "  -v \t\tVerbosely prints current generated trace (all variables).\n");
  fprintf(nusmv_stderr, "  -r \t\tSets picking mode to random (default is deterministic).\n");
  fprintf(nusmv_stderr, "  -i \t\tEnters simulation's interactive mode.\n");
  fprintf(nusmv_stderr, "  -a \t\tDisplays all the state variables (changed and unchanged)\n");
  fprintf(nusmv_stderr, "     \t\tin every step of an interactive session.\n");
  fprintf(nusmv_stderr, "     \t\tIt works only together with -i option.\n");
  fprintf(nusmv_stderr, "  -c \"constr\"\tSets constraint (simple expression) for the next steps.\n");
  fprintf(nusmv_stderr, "     \t\tNote: must be enclosed between double quotes \" \".\n");
  fprintf(nusmv_stderr, "  steps \tMaximum length of the path.\n");
  return(1);
}
