/**CFile***********************************************************************

  FileName    [simulate.c]

  PackageName [simulate]

  Synopsis    [Simulator routines]

  Description [This file contains functions for image computation, for state
	picking in a set (according to three different policies), for states display
	and constraints insertion.]

  Author      [Andrea Morichetti]

  Copyright   [
  This file is part of the ``simulate'' package of NuSMV version 2. 
  Copyright (C) 1998-2001 by CMU and ITC-irst. 

  NuSMV version 2 is free software; you can redistribute it and/or 
  modify it under the terms of the GNU Lesser General Public 
  License as published by the Free Software Foundation; either 
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful, 
  but WITHOUT ANY WARRANTY; without even the implied warranty of 
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public 
  License along with this library; if not, write to the Free Software 
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "simulateInt.h"
#include "error.h" /* for CATCH */

static char rcsid[] UTIL_UNUSED = "$Id: simulate.c,v 1.1.1.1 2003/02/06 19:01:18 flerda Exp $";

/**Variable********************************************************************

  Synopsis    [Stack and context for non-local goto inside simulator]

  Description [Stack and context for non-local goto inside simulator]

******************************************************************************/
#if SOLARIS
static sigjmp_buf simulate_jmp_buff;
#else
static jmp_buf simulate_jmp_buff;
#endif

static  void (*saved_simulate_sigterm)(int);
/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static bdd_ptr pick_state_rnd_or_det ARGS((bdd_ptr, Simulation_Mode));
static void show_next_states ARGS((bdd_ptr *, int, int));
static void simulate_sigterm ARGS((int));
static void print_trace ARGS((int, int, int));

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis           [Signal handler]

  Description        [SIGINT signal handler inside the simulator.]

  SideEffects        []

******************************************************************************/
static void simulate_sigterm(int sig) {
  signal(SIGINT, saved_simulate_sigterm);
  LONGJMP(simulate_jmp_buff, 1);
}

/**Function********************************************************************

  Synopsis           [Extracts a minterm from a given BDD]

  Description        [Picks a state from a set according to the Simulation_Mode
  (only Random or Deterministic: Interactive mode is not supported) entered: the
  first one sets to true those boolean variables which are indifferent for the
  actual BDD; the second one chooses those boolean values non deterministically.
  This is necessary in order to have unambiguous assignments for BDDs boolean
  variables.]

  SideEffects        []

  SeeAlso            [bdd_pick_one_state_rand bdd_pick_one_state]

******************************************************************************/
static bdd_ptr pick_state_rnd_or_det(bdd_ptr s, Simulation_Mode mode)
{
  if (mode != Random && mode != Deterministic) {
    fprintf(nusmv_stderr, "Warning: (pick_state_rnd_or_det) \"Deterministic\" mode will be used.");
    mode = Deterministic;
  }
  return(mode == Random ? bdd_pick_one_state_rand(dd_manager,s) :
	 bdd_pick_one_state(dd_manager,s));
}

/**Function********************************************************************

  Synopsis           [Stores a trace in the hash table and prints it.]

  Description        [Stores a trace in the hash table and prints it if the
  variable printyesno is true (this is set by the user via the command simulate
  options -p or -v).]

  SideEffects        []

  SeeAlso            []

******************************************************************************/
void store_and_print_trace(node_ptr p, boolean printyesno, int verbose)
{
  int state_no, trace_no;

  node_ptr current_state_label = current_state_label_get();

  if (opt_verbose_level_gt(options, 1) && printyesno == true) {
    fprintf(nusmv_stdout,
      "#####################################################\n");
    fprintf(nusmv_stdout,
      "######         Print Of Current Trace         #######\n");
    fprintf(nusmv_stdout,
      "#####################################################\n");
  }
  clear_print_hash();

  trace_no = trace_label_get_trace(current_state_label);
  state_no = trace_label_get_state(current_state_label);

  if (lookup_traces_hash(trace_mk_statelabel(trace_no, state_no+1)) == Nil) {
    state_no += 1;
    p = cdr(p);
  }
  else {
    state_no = 1;
    trace_no = get_trace_number();
    inc_trace_number();
  }

  set_indent_size(2);
  for( ; p != Nil; p = cdr(p)) {
    node_ptr label = trace_mk_statelabel(trace_no, state_no);

    bdd_ref((bdd_ptr)car(p));
    insert_traces_hash(label, car(p));
    if (printyesno == true) {
      fprintf(nusmv_stdout, "-> State ");
      print_node(nusmv_stdout, label);
      fprintf(nusmv_stdout, " <-\n");
      print_state((bdd_ptr)car(p), all_symbols, verbose);
    }
    state_no += 1;
  }
  reset_indent_size();
}
/**Function********************************************************************

  Synopsis           [Multiple step simulation]

  Description        [Multiple step simulation: loops n times over the choice of
  a state according to the picking policy given at command line. It returns a
  list of at least n+1 referenced states (the first one is always the "current
  state" from which any simulation must start). The obtained list can contain
  a minor number of states if there are no future states at some point.]

  SideEffects        []

  SeeAlso            [Simulate_ChooseOneState]

******************************************************************************/
node_ptr Simulate_MultipleSteps(Fsm_BddPtr fsm, bdd_ptr constraint,
                                Simulation_Mode mode, int n, int display_all)
{
  bdd_ptr current_state, invar_bdd;
  node_ptr result = Nil;
  int i = 1;

  invar_bdd = Compile_FsmBddGetInvar(fsm);
  current_state = current_state_bdd_get();
  /* we append the current state in first position of the trace */
  result = cons((node_ptr)current_state, result);
  while(i <= n) {
    bdd_ptr next_constr_set = (bdd_ptr)NULL;

    if (opt_verbose_level_gt(options, 1)) {
      switch (mode) {
      case Interactive:
        fprintf(nusmv_stderr,
                "********  Interactive mode Simulation step %d  *********\n",i);
        break;
      case Random:
	fprintf(nusmv_stderr,
		"**********  Random mode Simulation step %d  **********\n",i);
        break;
      case Deterministic:
	fprintf(nusmv_stderr,
		"*******  Deterministic mode Simulation step %d  *******\n",i);
        break;
      }
    }
    next_constr_set = Img_ImageFwd(fsm, current_state);
    bdd_and_accumulate(dd_manager, &next_constr_set, constraint);
    bdd_and_accumulate(dd_manager, &next_constr_set, invar_bdd);

    if (bdd_is_zero(dd_manager, next_constr_set)) {
      fprintf(nusmv_stderr, "No future state exists");
      fprintf(nusmv_stderr, (llength(result) == 1 ? ": trace not built.\n" : "."));
      fprintf(nusmv_stderr, " Simulation stopped at step %d.\n", i);
      bdd_free(dd_manager, next_constr_set);
      result = reverse(result);
      /* We don't free the current_state variable because the
	 list "result" must contain referenced states */
      return(result);
    }

    current_state = Simulate_ChooseOneState(next_constr_set, mode, display_all);

    if (current_state == (bdd_ptr)NULL || bdd_is_zero(dd_manager, current_state)) {
      fprintf(nusmv_stderr,
        "\nCan't find a future state. Simulation stopped at step %d.\n", i);
      if (current_state != (bdd_ptr)NULL) bdd_free(dd_manager, current_state);
      bdd_free(dd_manager, next_constr_set);
      result = reverse(result);
      return(result);
    }
    result = cons((node_ptr)current_state, result);
    /* we don't free the current_state variable because the final list "result"
       must contain referenced states: they will be freed after their insertion
       in the traces_hash table */
    i++;
    bdd_free(dd_manager, next_constr_set);
  }
  result = reverse(result);
  return(result);
}

/**Function********************************************************************

  Synopsis           [Chooses one state among future states]

  Description        [Chooses a state among future states depending on the
  given simulation policy (random, deterministic or interactive). In case of
	interactive simulation, the system stops and allows the user to pick
	a state	from a list of possible items. If the number of future states
	is too high, the system requires some further constraints to limit that
	number and will asks for them until the number of states is lower than
	an internal threshold. Entered expressions are accumulated in one big
	constraint used only in the actual step of the simulation. It will be
	discarded after a state will be chosen.]

  SideEffects        [A referenced state (BDD) is returned. NULL if failed.]

  SeeAlso            [Simulate_MultipleStep]

******************************************************************************/
bdd_ptr Simulate_ChooseOneState(bdd_ptr next_set, Simulation_Mode mode,
        int display_all)
{
  bdd_ptr result = (bdd_ptr)NULL;

  saved_simulate_sigterm = signal(SIGINT, simulate_sigterm);
  if (SETJMP(simulate_jmp_buff, 1) == 0) {
    switch(mode) {
    case Interactive: {
      boolean further = false;
      int i = 0;
      int choice = -1;
      int array_of_states_size = bdd_count_minterm(dd_manager, next_set, get_minterm_vars_dim());
      bdd_ptr *array_of_states     = (bdd_ptr *)NULL;
      bdd_ptr  further_constraints = bdd_one(dd_manager);
      node_ptr parsed_command      = Nil;
      bdd_ptr tmp_set = (bdd_ptr)NULL;
      
      /*
	There are 4 condition to be verified in order to accept new further
	constraints:
	
	1) entered expression must be a non-zero set;
	2) entered expression must be consistent with the accumulated constraints
	(i.e. the product (further /\ accumulated) must be non-zero;
	3) if (further /\ accumulated) is non-zero, it also must be non-zero the product
	(further /\ accumulated) /\ next_set of states
	4) cardinality of the set obtained from the last product must be < shown_states
	*/
      
      while (array_of_states_size > opt_shown_states_level(options)) {
	boolean done = false;
	
	further = true;
	fprintf(nusmv_stdout,
		"Too many (%d) future states to visualize. Please specify\n",
		array_of_states_size);
	
	while(done != true) {
	  char *strConstr[2];
	  char str[CONSTR_LENGTH];
	  
	  for (i=0; i<CONSTR_LENGTH; i++) str[i] = '\0';
	  fprintf(nusmv_stdout, "further constraints: ");
	  
	  /* Continue if receive a "good" string */
	  if ((fgets(str, CONSTR_LENGTH, nusmv_stdin) != (char)NULL) ||
	      (str[0] != '\n')) {
	    strConstr[0] = NIL(char);
	    strConstr[1] = util_strsav(str);
	    
	    CATCH {        
	      if (Parser_ReadCmdFromString(2, strConstr, "CONSTRAINT ",";\n",
					   &parsed_command)==0) {
		if ((parsed_command != Nil) &&
		    (node_get_type(parsed_command) == CONSTRAINT)) {
		  node_ptr constraints    = car(parsed_command);
		  add_ptr add_constraints = eval(constraints, Nil);
		  
		  bdd_ptr read_constr = add_to_bdd(dd_manager, add_constraints);
		  add_free(dd_manager, add_constraints);
		  if (bdd_is_zero(dd_manager, read_constr)) {
		    fprintf(nusmv_stderr,
			    "Entered constraints are equivalent to the empty set. Try again.\n");
		    bdd_free(dd_manager, read_constr);
		  }
		  else {/* entered expression is non-zero */
		    bdd_ptr tmp_constr = bdd_and(dd_manager, further_constraints, read_constr);
		    
		    bdd_free(dd_manager, read_constr);
		    if (bdd_is_zero(dd_manager, tmp_constr)) {
		      fprintf(nusmv_stderr,
			      "Entered expression is incompatible with old constraints. Try again.\n");
		      bdd_free(dd_manager, tmp_constr);
		    }
		    else {/* total constraints are non-zero */
		      tmp_set = bdd_and(dd_manager, tmp_constr, next_set);
		      
		      if (bdd_is_zero(dd_manager, tmp_set)) {
			fprintf(nusmv_stderr,
				"Set of future states is EMPTY: constraints too strong ? Try again.\n");
			bdd_free(dd_manager, tmp_constr);
			bdd_free(dd_manager, tmp_set);
		      }
		      else {/* next restricted set of states is non-zero */
			bdd_and_accumulate(dd_manager, &further_constraints, tmp_constr);
			
			bdd_free(dd_manager, tmp_constr);
			done = true;
			array_of_states_size =
			  bdd_count_minterm(dd_manager, tmp_set, get_minterm_vars_dim());
		      }
		    }
		  }
		}/* end if(parsed_command) */
		else
		  fprintf(nusmv_stderr,
			  "Parsing error: constraints must be \"simple_expressions\".\n");
	      } /* end if(Parser_ReadFromString) */
	      else
		fprintf(nusmv_stderr,
			"Parsing error: constraints must be \"simple_expressions\".\n");
	    } /* end CATCH */
	    FAIL { }
	  } /* end if (fgets) */
	} /* end while (done != true)*/
      } /* end while (array_of_states_size) */
      bdd_free(dd_manager, further_constraints);
      array_of_states = ALLOC(bdd_ptr, array_of_states_size);
      nusmv_assert(array_of_states != (bdd_ptr *)NULL);
      
      { /* chooses the set to pick from and fills the array */
	int res = bdd_pick_all_terms(dd_manager, ((tmp_set != (bdd_ptr)NULL) ?
						  tmp_set : next_set), minterm_vars, get_minterm_vars_dim(),
				     array_of_states, array_of_states_size);
	
	if (tmp_set != (bdd_ptr)NULL) bdd_free(dd_manager, tmp_set);
	if (res == 1) {
	  FREE(array_of_states);
	  return((bdd_ptr)NULL);
	}
      }
      show_next_states(array_of_states, array_of_states_size, display_all);
      
      /* Input of the choice */
      if (array_of_states_size > 1) {
	char line[CHOICE_LENGTH];
	
	for (i=0; i<CHOICE_LENGTH; i++) line[i] = '\0';
	fprintf(nusmv_stdout,
		"\nChoose a state from the above (0-%d): ",array_of_states_size-1);
	
	while((fgets(line, CONSTR_LENGTH, nusmv_stdin) != (char)NULL) ||
	      (line[0] != '\n')) {
	  if((sscanf(line, "%d", &choice) != 1) || (choice < 0) ||
	     (choice>array_of_states_size-1)) {
	    fprintf(nusmv_stdout,"Choose a state from the above (0-%d): ",
		    array_of_states_size-1);
	    continue;
	  }
	  else break;
	}
      }
      else {
	fprintf(nusmv_stdout,
		"\nThere's only one future state. Press Return to Proceed.");
	while ((choice = getc(nusmv_stdin)) != '\n') {;}
	choice = 0;
      }
      fprintf(nusmv_stdout, "\nChoosen state is: %d\n", choice);
      print_state(array_of_states[choice], all_symbols, display_all);
      fprintf(nusmv_stdout, "\n");
      result = bdd_dup(array_of_states[choice]);
      /* 
         The array returned by bdd_pick_all_terms has ref greater than 0.
         We have to dereference them
      */
      for (i=0; i < array_of_states_size; i++)
	bdd_free(dd_manager,array_of_states[i]);
      FREE(array_of_states);
      break;
    }
    default:
      result = pick_state_rnd_or_det(next_set, mode);
      break;
    } 
  } else {
    (void) fprintf(nusmv_stderr, "\n");
    /* signal(SIGINT, restore_signal); */
  }
  return(result);
}

/**Function********************************************************************

  Synopsis           [required]

  Description        [optional]

  SideEffects        [required]

  SeeAlso            [optional]

******************************************************************************/
int Simulate_CmdPickOneState(Fsm_BddPtr fsm, Simulation_Mode mode,
                             int display_all, char * strConstr)
{
  bdd_ptr init_bdd, invar_bdd;
  bdd_ptr initial_set;
  bdd_ptr chosen = (bdd_ptr)NULL;

  init_bdd = Compile_FsmBddGetInit(fsm);
  invar_bdd = Compile_FsmBddGetInvar(fsm);
  initial_set = bdd_and(dd_manager, init_bdd, invar_bdd);
  if (strConstr != NIL(char)) {
    node_ptr parsed = Nil;
    char * arg[1];

    arg[0] = NIL(char);
    arg[1] = strConstr;
    
    if (Parser_ReadCmdFromString(2, arg, "CONSTRAINT ", ";\n", &parsed) == 0) {
      CATCH {
        if ((parsed != Nil) && (node_get_type(parsed) == CONSTRAINT)) {
          node_ptr constraints = car(parsed);
          add_ptr add_constraints = eval(constraints, Nil);
          bdd_ptr bdd_constraints = add_to_bdd(dd_manager, add_constraints);
	    
          add_free(dd_manager, add_constraints);
          bdd_and_accumulate(dd_manager, &bdd_constraints, initial_set);
          chosen = Simulate_ChooseOneState(bdd_constraints, mode, display_all);
          bdd_free(dd_manager, bdd_constraints);
        }
        else {
          fprintf(nusmv_stderr, "Parsing error: constraints must be \"simple expressions\".\n");
          bdd_free(dd_manager, initial_set);
          return(1);
        }
      } /* end CATCH */
      FAIL {
        fprintf(nusmv_stderr, "Parsing error: constraints must be \"simple expressions\".\n");
        bdd_free(dd_manager, initial_set);
        return(1);
      }
    }
    else {
      fprintf(nusmv_stderr, "Parsing error: constraints must be \"simple expressions\".\n");
      bdd_free(dd_manager, initial_set);
      return(1);
    }
  }
  else {
    int i = bdd_count_minterm(dd_manager, initial_set, get_minterm_vars_dim());
      
    if (i == 0) {
      fprintf(nusmv_stderr, "The set of initial states is EMPTY. No state can be chosen.\n");
      bdd_free(dd_manager, initial_set);
      return(1);
    }
    chosen = Simulate_ChooseOneState(initial_set, mode, display_all);
  }
  bdd_free(dd_manager, initial_set);
  if (chosen == (bdd_ptr)NULL || bdd_is_zero(dd_manager, chosen)) {
    fprintf(nusmv_stderr, "Chosen state is the EMPTY set. No state has been chosen.\n");
    if (chosen != (bdd_ptr)NULL) bdd_free(dd_manager, chosen);
    return(1);
  }
  else {
    node_ptr label = trace_get_new_label();

    /* picked state is set as the 'current_state' and as the first state
       of a new trace */
    insert_traces_hash(label, (node_ptr)bdd_dup(chosen));
    current_state_set(chosen, label);
    bdd_free(dd_manager, chosen);
    inc_trace_number();
  }
  return(0);
}

/**Function********************************************************************

  Synopsis           [required]

  Description        [optional]

  SideEffects        [required]

  SeeAlso            [optional]

******************************************************************************/
int Simulate_CmdShowTraces(int trace, int verbose, boolean all, char * dbgFileName)
{
  /* Some traces exist, then one will be printed out according to the option
     or the number entered */

  if (all == false) {
    print_trace(trace, verbose, (dbgFileName != NIL(char)));
  }
  else {
    int c;
    int traceno = (get_trace_number() - 1);
    for (c = 1; c <= traceno; c++) {
      print_trace(c, verbose, (dbgFileName != NIL(char)));
    }
  }
  return(0);
}

/**Function********************************************************************

  Synopsis           [Prints an array of states on nusmv_stdout.]

  Description        [Prints an array of states on nusmv_stdout. If
  <tt>changes_only</tt> is 1, then only state variables which assume a
  different value from the previous printed one will be printed out.]

  SideEffects        []

******************************************************************************/
void show_next_states(bdd_ptr *array, int size, int changes_only)
{
  int j = 0;

  fprintf(nusmv_stdout,
    "***************  AVAILABLE FUTURE STATES  *************\n");
  clear_print_hash();
  for(j=0; j < size; j++){
    fprintf(nusmv_stdout, "\n");
    fprintf(nusmv_stdout, "%d) -------------------------\n", j);
    set_indent_size(2);
    print_state(array[j], all_symbols, changes_only);
    reset_indent_size();
  }
}

/**Function********************************************************************

  Synopsis           [required]

  Description        [optional]

  SideEffects        [required]

  SeeAlso            [optional]

******************************************************************************/
static void print_trace(int trace, int verbose, int flag) 
{
  int i = 1;
  node_ptr state = Nil;
  node_ptr label = trace_mk_statelabel(trace, i);

  clear_print_hash();
  state = lookup_traces_hash(label);
  if (flag == 1)
    fprintf (nusmv_stdout, "--- NuSMV Trace File --- ***** --- NuSMV Trace File ---\n");
  fprintf(nusmv_stdout,
          "################### Trace number: %d ###################\n", trace);
  while(state != Nil) {
    fprintf(nusmv_stdout, "-> State ");
    print_node(nusmv_stdout, label);
    fprintf(nusmv_stdout, " <-\n");
    print_state((bdd_ptr)state, all_symbols, verbose);
    ++i;
    label = trace_mk_statelabel(trace, i);
    state = lookup_traces_hash(label);
  }
}

