/**CFile***********************************************************************

  FileName    [satZChaffSolver.c]

  PackageName [sat]

  Synopsis    [Sat.ZChaffSolver module, the ZCHAFF solver wrapper]

  Description [This module contains an implementation of the SatSolver
  class, which maps the ZCHAFF solver specific sat solver.]

  SeeAlso     [satSolver]

  Author      [Roberto Cavada]

  Copyright   [
  This file is part of the ``sat'' package of NuSMV version 2.
  Copyright (C) 2000-2001 by ITC-irst and University of Trento.

  NuSMV version 2 is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "satZChaffSolver.h"
#include "satSolverInt.h"

#include "set.h"
#include "list.h"
#include "zchaff.h" /* the zchaff lib interface */ 



/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/**Struct**********************************************************************

  Synopsis    [This is the ZCHAFF solver wrapper]

  Description [This class extends the SatSolver class by using containment. 
  This is a private declaration, to actually use an instance of this class
  you must handle a Sat_ZChaffSolver_ptr.]

  SeeAlso     []

******************************************************************************/
typedef struct Sat_ZChaffSolver_TAG {
  Sat_SatSolver_ptr satSolver;
} Sat_ZChaffSolver;


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Declarations of internal functions                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static int 
sat_zchaff_initializer ARGS((Sat_SatSolver_ptr self, 
			     const Be_Cnf_ptr cnfProb));
static int  
sat_zchaff_quitter ARGS((Sat_SatSolver_ptr self));

static int  
sat_zchaff_caller ARGS((Sat_SatSolver_ptr self, 
			const Be_Cnf_ptr cnfProb, 
			Sat_SatResult_ptr result));
static int 
sat_zchaff_resolver ARGS((Sat_SatSolver_ptr self, 
			  Sat_SatResult_ptr result));

/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [ZChaffSolver constructor]

  Description [This constructor builds a ZChaff instance setting the 
  options values to their default values.]

  SideEffects []

  SeeAlso     [Sat_ZChaffSolver_Delete]

******************************************************************************/
Sat_ZChaffSolver_ptr Sat_ZChaffSolver_Create(void)
{
  Sat_ZChaffSolver_ptr self = NULL;
  char name[] = "ZCHAFF"; 
  SAT_Manager mng; /* the zchaff solver instance manager */

  self = ALLOC(Sat_ZChaffSolver, 1);
  nusmv_assert(self != NULL);

  /* instantiates the generic sat solver: */
  self->satSolver = Sat_SatSolver_Create(name, 
					 &sat_zchaff_initializer, 
					 &sat_zchaff_caller, 
					 &sat_zchaff_resolver, 
					 &sat_zchaff_quitter, 
					 NULL);

  return self;  
}


/**Function********************************************************************

  Synopsis    [ZChaffSolver destructor]

  Description [Call as soon as possible when you no longer need the given 
  instance]

  SideEffects []

  SeeAlso     [Sat_ZChaffSolver_Create]

******************************************************************************/
void Sat_ZChaffSolver_Delete(Sat_ZChaffSolver_ptr* self_ref)
{
  Sat_SatSolver_ptr solver;
  nusmv_assert(self_ref != NULL);
  if (*self_ref == NULL) return;

  solver = Sat_ZChaffSolver_GetSatSolver(*self_ref);
  Sat_SatSolver_Delete(&solver);
  FREE(*self_ref);
  *self_ref=NULL;  
}


/**Function********************************************************************

  Synopsis    [Converts the ZChaff-based instance to a generic sat solver]

  Description [Use this method to give the ZChaff instance to functions 
  which get a generic Sat solver]

  SideEffects []

  SeeAlso     []

******************************************************************************/
Sat_SatSolver_ptr Sat_ZChaffSolver_GetSatSolver(const Sat_ZChaffSolver_ptr self)
{
  return self->satSolver;
}


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/
static int 
sat_zchaff_initializer(Sat_SatSolver_ptr self, const Be_Cnf_ptr cnfProb)
{
  SAT_Manager mng = SAT_InitManager(); /* Solver inizialization */

  nusmv_assert(Sat_SatSolver_GetOptions(self) == NULL);

  /* options settings: */
#if 0 
  /* From the zchaff distribution: */
  SAT_SetMaxUnrelevance(mng, 20);
  SAT_SetMinClsLenForDelete(mng, 100);
  SAT_SetMaxConfClsLenAllowed(mng, 5000);

  /* randomness may help sometimes, by default, there is no randomness */
  SAT_SetRandomness (mng, 10);
  SAT_SetRandSeed (mng, -1);
#endif

  /* Sets the CNF problem into ZCHAFF: */
  SAT_SetNumVariables(mng, Be_Cnf_GetMaxVarIndex(cnfProb));

  /* Saves the manager so the caller can get it: */
  Sat_SatSolver_SetOptions(self, mng); 
  
  return 0;
}

static int sat_zchaff_quitter(Sat_SatSolver_ptr self)
{
  /* Gets the manager saved by the initializer: */
  SAT_Manager mng = (SAT_Manager) Sat_SatSolver_GetOptions(self);
  SAT_ReleaseManager(mng);   

  Sat_SatSolver_SetOptions(self, NULL);
  return 0;
}


static int sat_zchaff_caller(Sat_SatSolver_ptr self, 
			     const Be_Cnf_ptr cnfProb, 
			     Sat_SatResult_ptr result)
{
  int zchaffResult;
  lsGen    genCl, genLit;
  lsList   cl;
  int ret = 0; /* the return value */

  /* Gets the manager saved by the initializer: */
  SAT_Manager mng = (SAT_Manager) Sat_SatSolver_GetOptions(self);

  /* iterates across the clauses: */
  genCl = lsStart(Be_Cnf_GetClausesList(cnfProb));
  while (lsNext(genCl, (lsGeneric*) &cl, LS_NH) == LS_OK) {
    int var_idx;
    Set_t clause_vars;
    Set_t clause_lits;

    clause_vars = Set_MakeEmpty();
    clause_lits = Set_MakeEmpty();

    /* iterates across the clause variables indexes: */
    genLit = lsStart(cl);
    while (lsNext(genLit, (lsGeneric*) &var_idx, LS_NH) == LS_OK) {
      int sign;
      int lit;
      if( var_idx < 0) { 
	var_idx = -var_idx; 
	sign = 1; 
      }
      else sign = 0;
      lit = (var_idx << 1) + sign;

      /* this follows a ZCHAFF inner convention: */      
      clause_vars = Set_Union(clause_vars, 
			      Set_MakeSingleton((Set_Element_t)var_idx));
      clause_lits = Set_Union(clause_lits, 
			      Set_MakeSingleton((Set_Element_t)lit));      
    }
    lsFinish(genLit);
    
    /* commits this clause: */
    if ( (Set_GiveCardinality(clause_vars) != 0)
	 && (Set_GiveCardinality(clause_vars) 
	     == Set_GiveCardinality(clause_lits)) ) { 
      node_ptr list_lits = Set_Set2List(clause_lits);
      int array_len = llength(list_lits);
      int* array_lits = ALLOC(int, array_len); 
      int i;
      int lit; 

      for(i = 0; i < array_len; ++i) {
	nusmv_assert(list_lits != Nil);
	lit = (int) car(list_lits);
	array_lits[i] = lit;
	list_lits = cdr(list_lits);
      }
      SAT_AddClause(mng, array_lits, array_len);
      FREE(array_lits);
    }
    else {
      /* it contains vars of both polarity, so is automatically satisfied, 
	 just skip it */
    } 

    Set_ReleaseSet(clause_lits);
    Set_ReleaseSet(clause_vars);
  }
  lsFinish(genCl);

  
  zchaffResult = SAT_Solve(mng);

  switch(zchaffResult) {
  case UNSATISFIABLE:
    Sat_SatResult_SetUnsatisfiable(result);
    break;

  case SATISFIABLE:
    Sat_SatResult_SetSatisfiable(result);
    
    break;

  case TIME_OUT:
  case UNDETERMINED:
  case MEM_OUT:
  case ABORTED:
    Sat_SatResult_SetInternalError(result);
    ret = 1;
    break;

  default:
    nusmv_assert(false); /* no other cases are allowed */
  }

  return ret;
}



static int 
sat_zchaff_resolver(Sat_SatSolver_ptr self, 
		    Sat_SatResult_ptr result)
{
  /* Gets the manager saved by the initializer: */
  SAT_Manager mng = (SAT_Manager) Sat_SatSolver_GetOptions(self);
  lsList lsResult;
  int n_var = SAT_NumVariables(mng);
  int idx;

  nusmv_assert(Sat_SatResult_IsSatisfiable(result));

  lsResult = Sat_SatResult_GetListToCreatePropositionalModel(result);

  for (idx = 1; idx < n_var; ++idx) {
    int cnfIdx;
    int val = SAT_GetVarAsgnment(mng, idx);
    switch(val) {
    case -1: continue; /* does not store unassigned vars */
    case 0: cnfIdx = -idx;  break; /* negative polarity */
    case 1: cnfIdx = idx;  break; /* positive polarity */
    default:
      nusmv_assert(false); /* no other values should be provided */
    }
    lsNewEnd(lsResult, (lsGeneric)cnfIdx, LS_NH); /* appends the idx */
  }
    
  return 0;
}

