/**CHeaderFile*****************************************************************

  FileName    [satSolverInt.h]

  PackageName [sat]

  Synopsis    [Private interface for the generic sat solver type]

  Description [Pure base class Sat_Solver defines a set of private methods
  to be used by any derived class. See for example class Sat_SimSolver.]

  SeeAlso     []

  Author      [Roberto Cavada]

  Copyright   [
  This file is part of the ``sat'' package of NuSMV version 2.
  Copyright (C) 2000-2001 by ITC-irst and University of Trento.

  NuSMV version 2 is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#ifndef _SAT_SOLVER_INT_H
#define _SAT_SOLVER_INT_H

#include "sat.h"
#include "be.h"


/* ---------------------------------------------------------------------- */
/* All the following function return 1 if there is an error, 
   zero otherwise: */


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/**Struct**********************************************************************

  Synopsis    [Polymorphic method type automatically called to initialize 
  the solver before it could solve a problem]

  Description []

  SeeAlso     []

******************************************************************************/
typedef int  (*Sat_Initializer_fun)(Sat_SatSolver_ptr self, 
				    const Be_Cnf_ptr cnfProb);


/**Struct**********************************************************************

  Synopsis    [Polymorphic method type automatically called after the solver 
  solved a problem]

  Description []

  SeeAlso     []

******************************************************************************/
typedef int  (*Sat_Quitter_fun)(Sat_SatSolver_ptr self);


/**Struct**********************************************************************

  Synopsis    [Polymorphic method type which represents the actual 
  sat solver caller]

  Description [It gets a problem and returns a result as Sat_Result.]

  SeeAlso     []

******************************************************************************/
typedef int (*Sat_Caller_fun)(Sat_SatSolver_ptr self, 
			      const Be_Cnf_ptr cnfProb, 
			      Sat_SatResult_ptr satResult);


/* This function is called only if the result is satisfiable, to convert the 
   counterexample the solver returned into a complete SatResult */
/**Struct**********************************************************************

  Synopsis    [Converts the counterexample the solver produced into 
  a complete Sat_Result]

  Description [Polymorphic method type automatically called only if 
  the result is satisfiable]

  SeeAlso     []

******************************************************************************/
typedef int (*Sat_ModelMaker_fun)(Sat_SatSolver_ptr self,
				  Sat_SatResult_ptr satResult);


/**Struct**********************************************************************

  Synopsis    [The generic Sat solver]

  Description [To use an instance of class Sat_SatSolver you must handle 
  a Sat_SatSolver_ptr. Consider this class as a pure virtual base class.]

  SeeAlso     []

******************************************************************************/
typedef struct Sat_SatSolver_TAG {
  char* name;

  /* private virtual methods */
  Sat_Initializer_fun initializer;
  Sat_Caller_fun caller; /* actual sat invocation */
  Sat_ModelMaker_fun resolver; /* the trace conversion fun */
  Sat_Quitter_fun quitter; 

  /* solver-dependent options: */
  void* options; 

  /* for timing statistics: */
  long solvingTime;
} Sat_SatSolver;


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/

Sat_SatSolver_ptr Sat_SatSolver_Create 
ARGS((const char* name, 
      const Sat_Initializer_fun initializer, /* optional */
      const Sat_Caller_fun caller,           /* required */
      const Sat_ModelMaker_fun resolver,     /* optional */
      const Sat_Quitter_fun quitter,         /* optional */
      const void* opt));

void Sat_SatSolver_Delete ARGS((Sat_SatSolver_ptr* self_ref));

void Sat_SatSolver_SetOptions ARGS((const Sat_SatSolver_ptr self, void* opt));
				    

void* Sat_SatSolver_GetOptions ARGS((const Sat_SatSolver_ptr self));


#endif /* _SAT_SOLVER_INT_H */
