/**CFile***********************************************************************

  FileName    [satSolver.c]

  PackageName [sat]

  Synopsis    [Sat.Solver module]

  Description [This module keeps the generic sat solver container, while any 
  specific solver implementation must be located in a dedicated module.]

  SeeAlso     [satStruct, Bmc.bmcHLSat]

  Author      [Roberto Cavada]

  Copyright   [
  This file is part of the ``sat'' package of NuSMV version 2.
  Copyright (C) 2000-2001 by ITC-irst and University of Trento.

  NuSMV version 2 is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "satSolver.h"
#include "satInt.h" /* for 'options' and private Sat_Solver interface*/

#include <string.h>
#include "utils.h"
#include "opt.h"



/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Sat_SatSolver class constructor]

  Description        [This is a private method to be called from derivate
  classes only (probably from their constructors).
  Call Sat_SatSolver_Delete when you need no lonerg the created instance.]

  SideEffects        []

  SeeAlso            [Sat_SatSolver_Delete]
******************************************************************************/
Sat_SatSolver_ptr 
Sat_SatSolver_Create(const char* name, 
		     const Sat_Initializer_fun initializer, 
		     const Sat_Caller_fun caller, 
		     const Sat_ModelMaker_fun resolver,
		     const Sat_Quitter_fun quitter, 
		     const void* opt)

{
  Sat_SatSolver_ptr self = NULL; 

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Creating a SAT solver instance...  ");
  }

  self = ALLOC(Sat_SatSolver, 1);  nusmv_assert(self != NULL); 

  /* inits members: */
  if (name != NULL) {
    self->name = ALLOC(char, strlen(name)+1);
    nusmv_assert(self->name != NULL);
    strcpy(self->name, name);
  } 
  else {
    char defName[] = "Unknown";
    self->name = ALLOC(char, strlen(defName)+1);
    nusmv_assert(self->name != NULL);
    strcpy(self->name, defName);
  }
      
  self->initializer = initializer;
  self->caller = caller;
  self->resolver = resolver;
  self->quitter = quitter;
  self->options = (void*)opt;
  self->solvingTime = 0;  

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Created the solver '%s'\n", self->name);
  }

  return self;
}


/**Function********************************************************************

  Synopsis           [The Sat_SatSolver class destructor]

  Description        [This is a private method to be called from derivate
  classes only (probably from their destructors).]

  SideEffects        []

  SeeAlso            [Sat_SatSolver_Create]
******************************************************************************/
void Sat_SatSolver_Delete(Sat_SatSolver_ptr* self_ref)
{
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "The solver '%s' is going to be deleted...  ", 
	    (*self_ref)->name);
  }

  FREE((*self_ref)->name);
  FREE(*self_ref);
  *self_ref = NULL; /* invalidates the user's pointer */  

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Done\n");
  }
}


/**Function********************************************************************

  Synopsis           [Returns a readable name for the given sat solver 
  instance]

  Description        [The name is set by the constructor]

  SideEffects        []

  SeeAlso            []
******************************************************************************/
const char* 
Sat_SatSolver_GetName(const Sat_SatSolver_ptr self) { return self->name; }


/**Function********************************************************************

  Synopsis           [Makes the solver to solve the given problem in 
  cnf format]

  Description        [Result is returned via the satResult parameter]

  SideEffects        [satResult will change.]

  SeeAlso            []
******************************************************************************/
void Sat_SatSolver_Solve(const Sat_SatSolver_ptr self, 
			 const Be_Cnf_ptr cnfProb, 
			 Sat_SatResult_ptr satResult)
{
  int res = 0; /* the outcome of any virtual method call */

  /* we consider also the conversion time: */
  long start =  util_cpu_time(); 

  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Invoking solver '%s'...\n", 
	    Sat_SatSolver_GetName(self));
  }

  if (self->initializer != NULL) {
    res = self->initializer(self, cnfProb);
    if (res != 0) goto _sat_error;
  }

  nusmv_assert(self->caller != NULL); /* the only required field */
  res = self->caller(self, cnfProb, satResult);
  if (res != 0) goto _sat_error;

  if ((self->resolver != NULL) && Sat_SatResult_IsSatisfiable(satResult)) {
    res = self->resolver(self, satResult);
    if (res != 0) goto _sat_error;
  }


  if (self->quitter != NULL) {
    res = self->quitter(self);
    if (res != 0) goto _sat_error;
  }

  self->solvingTime = util_cpu_time() - start;
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Solver '%s' returned after %f secs, \n", 
	    Sat_SatSolver_GetName(self), 
	    Sat_SatSolver_GetLastSolvingTime(self)/1000.0);
  }
  return;

 _sat_error:
  if (opt_verbose_level_gt(options, 0)) {
    fprintf(nusmv_stderr, "Solver internal error\n");
  }    
  Sat_SatResult_SetInternalError(satResult);
}


/**Function********************************************************************

  Synopsis           [Returns the time spent by the sat solver 
  in the last solving]

  Description        []

  SideEffects        []

  SeeAlso            []
******************************************************************************/
long Sat_SatSolver_GetLastSolvingTime(const Sat_SatSolver_ptr self)
{
  return self->solvingTime;
}



/**Function********************************************************************

  Synopsis           [Sets the given private data]

  Description        [Use this private method from any derivate class in order
  to store useful data which can be accessed late via 
  Sat_SatSolver_GetOptions.]

  SideEffects        [self internal state will change]

  SeeAlso            [Sat_SatSolver_GetOptions]
******************************************************************************/
void Sat_SatSolver_SetOptions(Sat_SatSolver_ptr self, void* opt)
{
  self->options = opt;
}


/**Function********************************************************************

  Synopsis           [Gets the previusly set data]

  Description        [You can use this private method to access any derivate 
  class's data.]

  SideEffects        []

  SeeAlso            [Sat_SatSolver_SetOptions]
******************************************************************************/
void* Sat_SatSolver_GetOptions(const Sat_SatSolver_ptr self)
{
  return self->options;
}

/**AutomaticEnd***************************************************************/

