/**CFile***********************************************************************

  FileName    [satSimSolver.c]

  PackageName [sat]

  Synopsis    [Sat.SimSolver module, the SIM solver wrapper]

  Description [This module contains an implementation of the SatSolver
  class, which maps the SIM solver specific sat solver.]

  SeeAlso     [satSolver]

  Author      [Roberto Cavada]

  Copyright   [
  This file is part of the ``sat'' package of NuSMV version 2.
  Copyright (C) 2000-2001 by ITC-irst and University of Trento.

  NuSMV version 2 is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  NuSMV version 2 is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA.

  For more information of NuSMV see <http://nusmv.irst.itc.it>
  or email to <nusmv-users@irst.itc.it>.
  Please report bugs to <nusmv-users@irst.itc.it>.

  To contact the NuSMV development board, email to <nusmv@irst.itc.it>. ]

******************************************************************************/

#include "satSimSolver.h"
#include "satSolverInt.h"


#include "sim.h"
#include "list.h"


/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/

/**Struct**********************************************************************

  Synopsis    [This is the SIM solver wrapper]

  Description [This class extends the SatSolver class by using containment. 
  This is a private declaration, to actually use an instance of this class
  you must handle a Sat_SimSolver_ptr.]

  SeeAlso     []

******************************************************************************/
typedef struct Sat_SimSolver_TAG {
  Sat_SatSolver_ptr satSolver;
} Sat_SimSolver;



/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Declarations of internal functions                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/
static int 
sat_sim_initializer ARGS((Sat_SatSolver_ptr self, 
			  const Be_Cnf_ptr cnfProb));
static int  
sat_sim_quitter ARGS((Sat_SatSolver_ptr self));

static int  
sat_sim_caller ARGS((Sat_SatSolver_ptr self, 
		     const Be_Cnf_ptr cnfProb, 
		     Sat_SatResult_ptr result));
static int 
sat_sim_resolver ARGS((Sat_SatSolver_ptr self, 
		       Sat_SatResult_ptr result));


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis    [SimSolver constructor]

  Description [This constructor builds a SIM instance setting the 
  options values to their default values.]

  SideEffects []

  SeeAlso     [Sat_SimSolver_Delete]

******************************************************************************/
Sat_SimSolver_ptr Sat_SimSolver_Create(void)
{
  Sat_SimSolver_ptr self = NULL;
  int* pars = NULL; 

  self = ALLOC(Sat_SimSolver, 1);
  nusmv_assert(self != NULL);

  /* prepares the parameters set: */
  pars = Sim_ParamInit();
  nusmv_assert(pars != NULL);

  /* :TODO: Should this code be substituted by a configuration file, 
     or by a set of cmd-line options? */
  pars = Sim_ParamSet(pars, SIM_HEURISTICS, SIM_UNITIE_HEUR);
  pars = Sim_ParamSet(pars, SIM_RUN_TRACE, 0);
  pars = Sim_ParamSet(pars, SIM_INDEP_PROPS, 1);  
  
  /* instantiates the generic sat solver: */
  self->satSolver = Sat_SatSolver_Create("SIM", 
					 &sat_sim_initializer, 
					 &sat_sim_caller, 
					 &sat_sim_resolver, 
					 &sat_sim_quitter, 
					 pars);

  return self;
}


/**Function********************************************************************

  Synopsis    [SimSolver destructor]

  Description [Call as soon as possible when you no longer need the given 
  instance]

  SideEffects []

  SeeAlso     [Sat_SimSolver_Create]

******************************************************************************/
void Sat_SimSolver_Delete(Sat_SimSolver_ptr* self_ref)
{
  Sat_SatSolver_ptr solver; 
  nusmv_assert(self_ref != NULL);
  if (*self_ref == NULL) return;

  solver = Sat_SimSolver_GetSatSolver(*self_ref);
  Sat_SatSolver_Delete(&solver);
  FREE(*self_ref);
  *self_ref=NULL;
}


/**Function********************************************************************

  Synopsis    [Converts the SIM-based instance to a generic sat solver]

  Description [Use this method to give the SIM instance to functions 
  which get a generic Sat solver]

  SideEffects []

  SeeAlso     []

******************************************************************************/
Sat_SatSolver_ptr Sat_SimSolver_GetSatSolver(const Sat_SimSolver_ptr self)
{
  return self->satSolver;
}


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

static int sat_sim_initializer(Sat_SatSolver_ptr self, 
			       const Be_Cnf_ptr cnfProb)
{
  int* p = (int*)Sat_SatSolver_GetOptions(self);

  /* Sets the correct number of variables and clauses! */
  p = Sim_ParamSet(p, SIM_MAX_VAR_NUM, 
		   Be_Cnf_GetMaxVarIndex(cnfProb));
  
  p = Sim_ParamSet(p, SIM_MAX_CL_NUM, 
		   Be_Cnf_GetClausesNumber(cnfProb));

  /* Initializes Sim. */
  Sim_DllInit(p);
  return 0;
}

static int sat_sim_quitter(Sat_SatSolver_ptr self)
{
  /* Clean up SIM. */
  Sim_DllClear();
  return 0;
}


static int sat_sim_caller(Sat_SatSolver_ptr self, 
			   const Be_Cnf_ptr cnfProb, 
			   Sat_SatResult_ptr result)
{
  lsGen    genCl, genLit;
  lsList   cl;
  int      lit;

  /* Upload the clauses. */
  genCl = lsStart(Be_Cnf_GetClausesList(cnfProb));
  while (lsNext(genCl, (lsGeneric*) &cl, LS_NH) == LS_OK) {
    int clId = Sim_DllNewCl();

    genLit = lsStart(cl);    
    while (lsNext(genLit, (lsGeneric*) &lit, LS_NH) == LS_OK) {
      int res = Sim_DllAddLit(clId, lit);
      switch (res) {
      case 0 : 
	/* A tautology: go to the end of the clause */
	while (lsNext(genLit, (lsGeneric*) &lit, LS_NH) == LS_OK);
	/* Initiate a new clause. */
	clId = Sim_DllNewCl();
	break;

      case -1 :
	/* The program is gone berserk... */
	Sat_SatResult_SetInternalError(result);
	return 1;
      
      default :
	/* Everything is ok */
	break;
      }
    }
    lsFinish(genLit);
    /* Commits the clause */
    Sim_DllCommitCl(clId);
  }
  lsFinish(genCl);

  /* Uploads the independent variables */
  genLit = lsStart(Be_Cnf_GetVarsList(cnfProb));
  while (lsNext(genLit, (lsGeneric*) &lit, LS_NH) == LS_OK) {
    Sim_DllPropMakeIndep(lit);
  }
  lsFinish(genLit);


  /* Have Sim computed a solution? */
  if (Sim_DllSolve() == SIM_SAT) {
    Sat_SatResult_SetSatisfiable(result);
  } 
  else {
    Sat_SatResult_SetUnsatisfiable(result);
  }   

  return 0;
}

static int sat_sim_resolver(Sat_SatSolver_ptr self, 
			    Sat_SatResult_ptr result)
{
  int i;
  int* m;
  lsList lsResult;

  nusmv_assert(Sat_SatResult_IsSatisfiable(result));

  lsResult = Sat_SatResult_GetListToCreatePropositionalModel(result);
  m = Sim_DllGetStack();
  for (i = 1; i < m[0]; ++i) lsNewEnd(lsResult, (lsGeneric)m[i], LS_NH);
  return 0;
}

/**AutomaticEnd***************************************************************/

